(ns burningswell.db.weather.datasets
  (:refer-clojure :exclude [distinct find group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(def ^:private embedded-model
  '(case (is-null :weather.models.id) nil
         (json_build_object
          "id" :weather.models.id
          "name" :weather.models.name)))

(def ^:private embedded-variable
  '(case (is-null :weather.variables.id) nil
         (json_build_object
          "id" :weather.variables.id
          "name" :weather.variables.name)))

(defn- select-all [db & [opts]]
  (select db [:weather.datasets.id
              :weather.datasets.reference-time
              :weather.datasets.valid-time
              :weather.datasets.das
              :weather.datasets.dds
              :weather.datasets.dods
              :weather.datasets.created-at
              :weather.datasets.updated-at
              (as `(json_build_object
                    "model" ~embedded-model
                    "variable" ~embedded-variable)
                  :_embedded)]
    (from :weather.datasets)
    (join :weather.models.id :weather.datasets.model-id )
    (join :weather.variables.id :weather.datasets.variable-id)
    (paginate (:page opts) (:per-page opts))
    (order-by :weather.datasets.model-id
              :weather.datasets.variable-id
              :weather.datasets.valid-time)))

(s/defn all :- [WeatherDataset]
  "Return all weather datasets in `db`."
  [db :- Database]
  @(select-all db))

(s/defn by-id :- (s/maybe WeatherDataset)
  "Return the weather dataset in `db` by `id`."
  [db :- Database id :- s/Int]
  (first @(compose
           (select-all db)
           (where `(= :weather.datasets.id (cast ~id :integer))))))

(defn- natural-key-clause
  "Return the natural key clause for `dataset`."
  [dataset]
  `(or (= :weather.datasets.id ~(:id dataset))
       (and (= :weather.datasets.model-id
               ~(-> dataset :_embedded :model :id))
            (= :weather.datasets.variable-id
               ~(-> dataset :_embedded :variable :id))
            (= :weather.datasets.valid-time
               ~(:valid-time dataset)))))

(s/defn find :- (s/maybe WeatherDataset)
  "Return the weather dataset in `db` by `id`."
  [db :- Database dataset]
  (first @(compose
           (select-all db)
           (where (natural-key-clause dataset)))))

(defn- row [dataset]
  (-> (select-keys dataset [:reference-time :valid-time :das :dds :dods])
      (assoc :model-id (-> dataset :_embedded :model :id))
      (assoc :variable-id (-> dataset :_embedded :variable :id))))

(s/defn delete
  "Delete the weater `dataset` from `db`."
  [db :- Database dataset :- WeatherDataset]
  (->> @(sql/delete db :weather.datasets
          (where `(= :weather.datasets.id
                     ~(:id dataset))))
       first :count))

(s/defn insert :- WeatherDataset
  "Insert the weater `dataset` into `db`."
  [db :- Database dataset]
  (->> @(sql/insert db :weather.datasets []
          (values [(row dataset)])
          (returning :id))
       first :id (by-id db)))

(s/defn update :- (s/maybe WeatherDataset)
  "Update the weater `dataset` in `db`."
  [db :- Database dataset]
  (some->> @(sql/update db :weather.datasets
              (row dataset)
              (where (natural-key-clause dataset))
              (returning :id))
           first :id (by-id db)))

(s/defn save :- WeatherDataset
  "Save the weather `dataset` to `db`."
  [db :- Database datset]
  (or (update db datset)
      (insert db datset)))

(s/defn exists? :- s/Bool
  "Return true if the weather `dataset` exists in `db`."
  [db :- Database dataset]
  (not (nil? (find db dataset))))
