(ns burningswell.db.weather.variables
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all [db & [opts]]
  (select db [:weather.variables.id
              :weather.variables.name
              :weather.variables.description
              :weather.variables.unit
              :weather.variables.created-at
              :weather.variables.updated-at]
    (from :weather.variables)
    (paginate (:page opts) (:per-page opts))
    (order-by :weather.variables.name)))

(s/defn all :- [WeatherVariable]
  "Return all weather variables in `db`."
  [db :- Database]
  @(select-all db))

(s/defn by-id :- (s/maybe WeatherVariable)
  "Return the weather variable in `db` by `id`."
  [db :- Database id :- s/Int]
  (first @(compose
           (select-all db)
           (where `(= :weather.variables.id (cast ~id :integer))))))

(s/defn by-name :- (s/maybe WeatherVariable)
  "Return the weather variable in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :weather.variables.name ~name)))))

(s/defn by-names :- [WeatherVariable]
  "Return the weather variables in `db` with a name in `names`."
  [db :- Database names :- [s/Str]]
  @(compose
    (select-all db)
    (where `(in :weather.variables.name ~(seq names)))))

(s/defn by-names-or-all :- [WeatherVariable]
  "Return the weather variables in `db` with a name in `names`, or all
  weather variables if `names` is empty."
  [db :- Database names :- [s/Str]]
  (if (empty? names)
    (all db) (by-names db names)))

(s/defn htsgwsfc :- (s/maybe WeatherVariable)
  "Return the significant height of combined wind waves and swell
  variable."
  [db :- Database]
  (by-name db "htsgwsfc"))
