(ns burningswell.db.continents
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [clojure.string :as str]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [no.en.core :refer [parse-integer]]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all [db & [opts]]
  (select db [:continents.id
              :continents.name
              :continents.code
              :continents.airport-count
              :continents.country-count
              :continents.port-count
              :continents.region-count
              :continents.spot-count
              :continents.user-count
              :continents.created-at
              :continents.updated-at]
    (from :continents)
    (fulltext (:query opts) :continents.name)
    (order-by-distance :continents.geom (:location opts))
    (order-by :continents.name)
    (paginate (:page opts) (:per-page opts))))

(s/defn all :- [Continent]
  "Return all continents in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-code :- (s/maybe Continent)
  "Return the continent in `db` by `code`."
  [db :- Database code :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :continents.code
                      ~(str/lower-case code))))))

(s/defn by-id :- (s/maybe Continent)
  "Return the continent in `db` by `id`."
  [db :- Database id :- s/Int]
  (first @(compose
           (select-all db)
           (where `(= :continents.id (cast ~id :integer))))))

(s/defn by-name :- (s/maybe Continent)
  "Return the continent in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :continents.name ~name)))))

(s/defn by-country :- (s/maybe Continent)
  "Load the region of `country` from `db`."
  [db :- Database country :- Country]
  (by-id db (-> country :_embedded :continent :id)))

(s/defn by-spot :- (s/maybe Continent)
  "Return the continent in `db` by `name`."
  [db :- Database spot :- Spot]
  (let [country-id (-> spot :_embedded :country :id)]
    (first @(compose
             (select-all db)
             (join :countries.continent-id :continents.id)
             (where `(= :countries.id ~country-id))))))

(defn- row [continent]
  (select-keys continent [:name :code]))

(s/defn delete
  "Delete `continent` from `db`."
  [db :- Database continent :- Continent]
  (->> @(sql/delete db :continents
          (where `(= :continents.id
                     ~(:id continent))))
       first :count))

(s/defn insert :- Continent
  "Insert `continent` into `db`."
  [db :- Database continent]
  (->> @(sql/insert db :continents []
          (values [(row continent)])
          (returning :id))
       first :id (by-id db)))

(s/defn update :- Continent
  "Update `continent` in `db`."
  [db :- Database continent]
  (->> @(sql/update db :continents (row continent)
                    (where `(= :continents.id
                               ~(:id continent)))
                    (returning :id))
       first :id (by-id db)))

(s/defn asia :- (s/maybe Continent)
  "Find the continent Asia."
  [db :- Database]
  (by-name db "Asia"))

(s/defn europe :- (s/maybe Continent)
  "Find the continent Europe."
  [db :- Database]
  (by-name db "Europe"))

(s/defn north-america :- (s/maybe Continent)
  "Find the the continent North America."
  [db :- Database]
  (by-name db "North America"))

(s/defn update-counters
  "Update the counters of all continents."
  [db :- Database]
  (->> [(select db ['(update-continent-airport-count)])
        (select db ['(update-continent-country-count)])
        (select db ['(update-continent-port-count)])
        (select db ['(update-continent-region-count)])
        (select db ['(update-continent-spot-count)])
        (select db ['(update-continent-user-count)])]
       (map (comp first deref))
       (apply merge)))
