(ns burningswell.db.countries
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.photos :as photos]
            [burningswell.db.util :as util :refer :all]
            [clojure.string :as str]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [geo.postgis :as geo]
            [no.en.core :refer [parse-integer]]
            [schema.core :as s])
  (:import sqlingvo.db.Database
           org.postgis.Point))

(defn- row [country]
  (let [continent (-> country :_embedded :continent)]
    (-> (select-keys country
                     [:area
                      :fips-code
                      :iso-3166-1-alpha-2
                      :iso-3166-1-alpha-3
                      :iso-3166-1-numeric
                      :location
                      :name
                      :phone-prefix
                      :population])
        (assoc :continent-id (:id continent)))))

(defn- select-all [db & [opts]]
  (let [{:keys [bounding-box distance location min-spots]} opts]
    (select db [:countries.id
                :countries.name
                :countries.iso-3166-1-alpha-2
                :countries.iso-3166-1-alpha-3
                :countries.iso-3166-1-numeric
                :countries.fips-code
                :countries.phone-prefix
                :countries.area
                (as '(cast :countries.location :geometry) :location)
                :countries.population
                :countries.airport-count
                :countries.port-count
                :countries.region-count
                :countries.spot-count
                :countries.user-count
                :countries.created-at
                :countries.updated-at
                (as `(json_build_object
                      "continent" (json-embed-continent :continents)
                      "photo" ~embedded-photo)
                    :_embedded)]
      (from :countries)
      (join :continents.id :countries.continent-id)
      (join :photos.id :countries.photo-id :type :left)
      (fulltext (:query opts) :countries.name)
      (within-bounding-box :countries.geom bounding-box)
      (within-distance-to :countries.geom location distance)
      (when min-spots
        (where `(>= :countries.spot-count (cast ~min-spots :integer)) :and))
      (order-by-distance :countries.geom location)
      (order-by :countries.name)
      (paginate (:page opts) (:per-page opts)))))

(s/defn all :- [Country]
  "Return all countries in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-id :- (s/maybe Country)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :countries.id (cast ~id :integer))))))

(s/defn by-ids :- [Country]
  "Return all countries in `db` by the list of `ids`."
  [db :- Database ids :- [s/Num] & [opts]]
  @(compose
    (select-all db opts)
    (where `(in :countries.id ~ids))))

(s/defn by-iso-3166-1-alpha-2 :- (s/maybe Country)
  "Return the country in `db` by `iso-3166-1-alpha-2` country code."
  [db :- Database iso-3166-1-alpha-2 :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :countries.iso-3166-1-alpha-2
                      ~(str/lower-case iso-3166-1-alpha-2))))))

(s/defn by-name :- (s/maybe Country)
  "Return the country in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :countries.name ~name)))))

(s/defn by-location :- [Country]
  "Return the country in `db` near `location`."
  [db :- Database location :- Point]
  (let [location (geo/geometry location)]
    @(compose
      (select-all db)
      (where `(st_intersects :countries.geom (cast ~location :geography)))
      (order-by `(st_distance (cast ~location :geography) :countries.geom)))))

(s/defn within-distance :- [Country]
  "Return the countries in `db` within `distance` in km to `location`."
  [db :- Database location :- Point distance :- s/Num & [opts]]
  @(compose
    (select-all db (assoc opts :location location))
    (within-distance-to :countries.geom location distance)))

(s/defn closest :- (s/maybe Country)
  "Find the closest country to `location`."
  [db :- Database location :- Point & [distance :- s/Num]]
  (or (first (within-distance db location (or distance 20)))
      (first (by-location db location))))

(s/defn by-spot :- (s/maybe Country)
  "Load the country of `spot` from `db`."
  [db :- Database spot :- Spot]
  (some->> spot :_embedded :country :id
           (by-id db)))

(s/defn delete
  "Delete `country` from `db`."
  [db :- Database country :- Country]
  (->> @(sql/delete db :countries
          (where `(= :countries.id
                     ~(:id country))))
       first :count))

(s/defn in-continent
  "Returns all countries in `continent`."
  [db :- Database continent :- Continent & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :countries.continent-id
               ~(:id continent)))))

(s/defn insert
  "Insert `country` into `db`."
  [db :- Database country]
  (->> @(sql/insert db :countries []
          (values [(row country)])
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `country` in `db`."
  [db :- Database country]
  (->> @(sql/update db :countries
          (row country)
          (where `(= :countries.id
                     ~(:id country)))
          (returning :id))
       first :id (by-id db)))

(s/defn germany :- (s/maybe Country)
  "Find the country Germany."
  [db :- Database]
  (by-iso-3166-1-alpha-2 db "de"))

(s/defn indonesia :- (s/maybe Country)
  "Find the country Indonesia."
  [db :- Database]
  (by-iso-3166-1-alpha-2 db "id"))

(s/defn spain :- (s/maybe Country)
  "Find the country Spain."
  [db :- Database]
  (by-iso-3166-1-alpha-2 db "es"))

(s/defn united-states :- (s/maybe Country)
  "Find the country United States."
  [db :- Database]
  (by-iso-3166-1-alpha-2 db "us"))

(s/defn update-counters
  "Update the counters of all countries."
  [db :- Database]
  (->> [(select db ['(update-country-airport-count)])
        (select db ['(update-country-port-count)])
        (select db ['(update-country-region-count)])
        (select db ['(update-country-spot-count)])
        (select db ['(update-country-user-count)])]
       (map (comp first deref))
       (apply merge)))

(s/defn top-photos
  "Return the top photo for each country."
  [db :- Database]
  (select db (distinct
              [:photos-countries.country-id
               :photos-countries.photo-id]
              :on [:photos-countries.country-id] )
    (from :photos-countries)
    (join :photos.id :photos-countries.photo-id)
    (order-by :photos-countries.country-id
              (desc :photos.created-at))))

(s/defn update-photos
  "Update the photos of all countries."
  [db :- Database]
  (sql/update db :countries
    {:photo-id :top-photos.photo-id}
    (from (as (top-photos db) :top-photos))
    (where `(= :countries.id :top-photos.country-id))))

(s/defn assoc-photo :- [Country]
  "Assoc the photo and their images to all `countries`."
  [db :- Database countries :- [Country]]
  (let [photos (zip-by-id (photos/by-countries db countries {:images true}))]
    (map #(->> (get photos (-> % :_embedded :photo :id))
               (assoc-in % [:_embedded :photo]))
         countries)))

(s/defn as-png :- s/Any
  "Return the `country` as PNG."
  [db :- Database country :- Country & [opts]]
  (let [{:keys [width height color]} opts]
    (->> @(select db [(as (util/geometry-as-png width height color) :png)]
            (from :countries)
            (where `(= :countries.id ~(:id country))))
         first :png)))

(s/defn geometry :- GeoJSON
  "Return the geometry of `country` as GeoJSON data structure."
  [db :- Database country :- Country]
  (geojson-by-column db :countries :geom :id (:id country)))
