(ns burningswell.db.countries
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]
            [datumbazo.record :as record]))

(t/deftable countries
  "The countries table."
  (t/column :airport-count :integer)
  (t/column :area :integer)
  (t/column :continent-id :integer :not-null? true :references :continents.id)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :fips-code :citext)
  (t/column :geom :geography :geometry :multi-polygon :srid 4326)
  (t/column :geonames-id :integer)
  (t/column :id :integer :primary-key? true)
  (t/column :iso-3166-1-alpha-2 :citext :not-null? true :unique? true)
  (t/column :iso-3166-1-alpha-3 :citext :not-null? true :unique? true)
  (t/column :iso-3166-1-numeric :integer :not-null? true :unique? true)
  (t/column :location :geography :point :multi-polygon :srid 4326)
  (t/column :name :citext :not-null? true :unique? true)
  (t/column :phone-prefix :citext)
  (t/column :photo-id :integer)
  (t/column :population :integer :not-null? true)
  (t/column :port-count :integer)
  (t/column :region-count :integer)
  (t/column :spot-count :integer)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :user-count :integer))

(defn- select-expr [db]
  (record/select-columns db Country))

(defn germany
  "Find the country Germany."
  [db]
  (by-iso-3166-1-alpha-2 db "de"))

(s/fdef germany
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::country))

(defn indonesia
  "Find the country Indonesia."
  [db]
  (by-iso-3166-1-alpha-2 db "id"))

(s/fdef indonesia
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::country))

(defn spain
  "Find the country Spain."
  [db]
  (by-iso-3166-1-alpha-2 db "es"))

(s/fdef spain
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::country))

(defn united-states
  "Find the country United States."
  [db]
  (by-iso-3166-1-alpha-2 db "us"))

(s/fdef united-states
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::country))

(defn within-distance
  "Return the countries in `db` within `distance` in km to `location`."
  [db location distance & [opts]]
  @(sql/select db (select-expr db)
     (sql/from :countries)
     (util/within-distance-to :geom location distance)
     (util/order-by-distance :geom location)))

(s/fdef within-distance
  :args (s/cat :db sql/db? :location ::location :distance number? :opts (s/? map?))
  :ret (s/coll-of ::country))

(defn closest
  "Find the closest country to `location`."
  [db location & [distance]]
  (or (first (within-distance db location (or distance 20)))
      (first (by-location db location))))

(s/fdef closest
  :args (s/cat :db sql/db? :location ::location :distance (s/? number?))
  :ret (s/coll-of ::country))

(defn by-spot
  "Returns the country of `spot` from `db`."
  [db spot]
  (some->> spot :country-id (by-id db)))

(s/fdef by-spot
  :args (s/cat :db sql/db? :spot :burningswell.db.spots/spot)
  :ret (s/nilable ::country))

;; (defn- row [country]
;;   (let [continent (-> country :_embedded :continent)]
;;     (-> (select-keys country
;;                      [:area
;;                       :fips-code
;;                       :iso-3166-1-alpha-2
;;                       :iso-3166-1-alpha-3
;;                       :iso-3166-1-numeric
;;                       :location
;;                       :name
;;                       :phone-prefix
;;                       :population])
;;         (assoc :continent-id (:id continent)))))

;; (defn- select-all [db & [opts]]
;;   (let [{:keys [bounding-box distance location min-spots]} opts]
;;     (select db [:countries.id
;;                 :countries.name
;;                 :countries.iso-3166-1-alpha-2
;;                 :countries.iso-3166-1-alpha-3
;;                 :countries.iso-3166-1-numeric
;;                 :countries.fips-code
;;                 :countries.phone-prefix
;;                 :countries.area
;;                 (as '(cast :countries.location :geometry) :location)
;;                 :countries.population
;;                 :countries.airport-count
;;                 :countries.port-count
;;                 :countries.region-count
;;                 :countries.spot-count
;;                 :countries.user-count
;;                 :countries.created-at
;;                 :countries.updated-at
;;                 (as `(json_build_object
;;                       "continent" (json-embed-continent :continents)
;;                       "photo" ~embedded-photo)
;;                     :_embedded)]
;;             (from :countries)
;;             (join :continents.id :countries.continent-id)
;;             (join :photos.id :countries.photo-id :type :left)
;;             (fulltext (:query opts) :countries.name)
;;             (within-bounding-box :countries.geom bounding-box)
;;             (within-distance-to :countries.geom location distance)
;;             (when min-spots
;;               (where `(>= :countries.spot-count (cast ~min-spots :integer)) :and))
;;             (order-by-distance :countries.geom location)
;;             (order-by :countries.name)
;;             (paginate (:page opts) (:per-page opts)))))

;; (s/defn all :- [Country]
;;   "Return all countries in `db`."
;;   [db :- Database & [opts]]
;;   @(select-all db opts))

;; (s/defn by-id :- (s/maybe Country)
;;   "Return the country in `db` by `id`."
;;   [db :- Database id :- s/Num]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :countries.id (cast ~id :integer))))))

;; (s/defn by-ids :- [Country]
;;   "Return all countries in `db` by the list of `ids`."
;;   [db :- Database ids :- [s/Num] & [opts]]
;;   @(compose
;;     (select-all db opts)
;;     (where `(in :countries.id ~ids))))

;; (s/defn by-iso-3166-1-alpha-2 :- (s/maybe Country)
;;   "Return the country in `db` by `iso-3166-1-alpha-2` country code."
;;   [db :- Database iso-3166-1-alpha-2 :- s/Str]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :countries.iso-3166-1-alpha-2
;;                       ~(str/lower-case iso-3166-1-alpha-2))))))

;; (s/defn by-name :- (s/maybe Country)
;;   "Return the country in `db` by `name`."
;;   [db :- Database name :- s/Str]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :countries.name ~name)))))

;; (s/defn by-location :- [Country]
;;   "Return the country in `db` near `location`."
;;   [db :- Database location :- Point]
;;   (let [location (geo/geometry location)]
;;     @(compose
;;       (select-all db)
;;       (where `(st_intersects :countries.geom (cast ~location :geography)))
;;       (order-by `(st_distance (cast ~location :geography) :countries.geom)))))

;; (s/defn in-continent
;;   "Returns all countries in `continent`."
;;   [db :- Database continent :- Continent & [opts]]
;;   @(compose
;;     (select-all db opts)
;;     (where `(= :countries.continent-id
;;                ~(:id continent)))))

;; (s/defn insert
;;   "Insert `country` into `db`."
;;   [db :- Database country]
;;   (->> @(sql/insert db :countries []
;;                     (values [(row country)])
;;                     (returning :id))
;;        first :id (by-id db)))

;; (s/defn update
;;   "Update `country` in `db`."
;;   [db :- Database country]
;;   (->> @(sql/update db :countries
;;                     (row country)
;;                     (where `(= :countries.id
;;                                ~(:id country)))
;;                     (returning :id))
;;        first :id (by-id db)))

;; (s/defn germany :- (s/maybe Country)
;;   "Find the country Germany."
;;   [db :- Database]
;;   (by-iso-3166-1-alpha-2 db "de"))

;; (s/defn indonesia :- (s/maybe Country)
;;   "Find the country Indonesia."
;;   [db :- Database]
;;   (by-iso-3166-1-alpha-2 db "id"))

;; (s/defn spain :- (s/maybe Country)
;;   "Find the country Spain."
;;   [db :- Database]
;;   (by-iso-3166-1-alpha-2 db "es"))

;; (s/defn united-states :- (s/maybe Country)
;;   "Find the country United States."
;;   [db :- Database]
;;   (by-iso-3166-1-alpha-2 db "us"))

;; (s/defn update-counters
;;   "Update the counters of all countries."
;;   [db :- Database]
;;   (->> [(select db ['(update-country-airport-count)])
;;         (select db ['(update-country-port-count)])
;;         (select db ['(update-country-region-count)])
;;         (select db ['(update-country-spot-count)])
;;         (select db ['(update-country-user-count)])]
;;        (map (comp first deref))
;;        (apply merge)))

;; (s/defn top-photos
;;   "Return the top photo for each country."
;;   [db :- Database]
;;   (select db (distinct
;;               [:photos-countries.country-id
;;                :photos-countries.photo-id]
;;               :on [:photos-countries.country-id] )
;;           (from :photos-countries)
;;           (join :photos.id :photos-countries.photo-id)
;;           (order-by :photos-countries.country-id
;;                     (desc :photos.created-at))))

;; (s/defn update-photos
;;   "Update the photos of all countries."
;;   [db :- Database]
;;   (sql/update db :countries
;;               {:photo-id :top-photos.photo-id}
;;               (from (as (top-photos db) :top-photos))
;;               (where `(= :countries.id :top-photos.country-id))))

;; (s/defn assoc-photo :- [Country]
;;   "Assoc the photo and their images to all `countries`."
;;   [db :- Database countries :- [Country]]
;;   (let [photos (zip-by-id (photos/by-countries db countries {:images true}))]
;;     (map #(->> (get photos (-> % :_embedded :photo :id))
;;                (assoc-in % [:_embedded :photo]))
;;          countries)))

(defn as-png
  "Return the `country` as PNG."
  [db country & [opts]]
  (let [{:keys [width height color]} opts]
    (->> @(sql/select db [(sql/as (util/geometry-as-png width height color) :png)]
            (sql/from :countries)
            (sql/where `(= :countries.id ~(:id country))))
         first :png)))

;; (s/defn geometry :- GeoJSON
;;   "Return the geometry of `country` as GeoJSON data structure."
;;   [db :- Database country :- Country]
;;   (geojson-by-column db :countries :geom :id (:id country)))

;; (defn exists?
;;   "Returns true if `country` exists in `db`, otherwise false."
;;   [db country]
;;   (not (empty? @(sql/select db [:id]
;;                             (sql/from :countries)
;;                             (sql/where `(= :id ~(:id country)))))))

;; (defn select-nearby-location
;;   "Select all countries nearby `location`."
;;   [db location & [opts]]
;;   {:pre [(sql/db? db)]}
;;   (select-all db (assoc opts :location location)))

;; (defn nearby-country
;;   "Returns all countries nearby `country`."
;;   [db country & [opts]]
;;   {:pre [(sql/db? db)]}
;;   @(compose
;;     (select-nearby-location db (:location country) opts)
;;     (where `(<> :countries.id ~(:id country)) :and)))
