(ns burningswell.db.search
  (:require [burningswell.db.util :refer :all]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [inflections.core :as infl]))

(defn refresh-views
  "Refresh the search views."
  [db]
  @(sql/refresh-materialized-view db :search.autocomplete
     (sql/concurrently true)))

(s/fdef refresh-views
  :args (s/cat :db sql/db?))

(defn- rank
  "Returns the search rank expression."
  [query]
  (sql/as `(cast
            (ts_rank_cd
             (to_tsvector :name)
             (to_tsquery ~query))
            :decimal)
          :rank))

(defn- result-type
  "Returns the result type expression."
  [table]
  (sql/as (name (infl/singular table)) :type))

(defn- search-table
  "Search the given `table` in `db`."
  [db table & [{:keys [query limit offset order-by]}]]
  (sql/select db [:id (result-type table) (rank query)]
    (sql/from table)
    (fulltext query :name)
    (sql/offset offset)
    (sql/limit limit)
    (apply sql/order-by order-by)))

(defn search
  "Return all comments in `db`."
  [db & [{:keys [limit offset query] :as opts}]]
  (let [last-opts (assoc opts :order-by [:rank])
        opts (dissoc opts :limit :offset :order-by)]
    (->> @(sql/union
           (search-table db :continents opts)
           (search-table db :countries opts)
           (search-table db :regions opts)
           (search-table db :spots last-opts))
         (map #(update-in % [:type] keyword)))))

(s/def :burningswell.db.search.opts/limit (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/offset (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/query (s/nilable string?))

(s/def ::search-opts
  (s/keys :opt-un [:burningswell.db.search.opts/limit
                   :burningswell.db.search.opts/offset
                   :burningswell.db.search.opts/query]))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? (s/nilable ::search-opts))))
