(ns burningswell.db.generators
  (:require [clojure.string :as str]
            [clojure.test.check.generators :as gen]))

(defn string-alphanumeric-fixed
  "Generate instants of time."
  [n]
  (gen/fmap str/join (gen/vector gen/char-alpha n)))

(def inst
  "Generate instants of time."
  (gen/fmap #(java.util.Date. %) gen/s-pos-int))

(def continents
  "Generate continents."
  (gen/hash-map
   :airport-count gen/s-pos-int
   :code (string-alphanumeric-fixed 2)
   :country-count gen/s-pos-int
   :created-at inst
   :id gen/s-pos-int
   :name gen/string-alphanumeric
   :port-count gen/s-pos-int
   :region-count gen/s-pos-int
   :spot-count gen/s-pos-int
   :updated-at inst
   :user-count gen/s-pos-int))

(defn embedded-continent
  "Returns a generator for an embedded `continent`."
  [continent]
  (gen/fmap #(select-keys % [:id :code :name]) continent))

(defn countries
  "Generate countries."
  [& {:keys [continent]}]
  (gen/hash-map
   :airport-count gen/s-pos-int
   :area gen/s-pos-int
   :fips-code (string-alphanumeric-fixed 2)
   :iso-3166-1-alpha-2 (string-alphanumeric-fixed 2)
   :iso-3166-1-alpha-3 (string-alphanumeric-fixed 3)
   :iso-3166-1-numeric gen/s-pos-int
   :created-at inst
   :id gen/s-pos-int
   :name gen/string-alphanumeric
   :phone-prefix gen/s-pos-int
   :population gen/s-pos-int
   :port-count gen/s-pos-int
   :region-count gen/s-pos-int
   :spot-count gen/s-pos-int
   :updated-at inst
   :user-count gen/s-pos-int
   :_embedded
   (gen/hash-map
    :continent
    (embedded-continent
     (if continent
       (gen/return continent)
       continents)))))

(defn sample
  "Return a sequence of `num-samples` (default 10) realized values
  from `generator`."
  [generator & [num-samples]]
  (gen/sample generator (or num-samples 10)))

(defn example
  "Return an example from `generator`."
  [generator]
  (last (sample generator)))
