(ns burningswell.db.photos
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.record :as record]
            [datumbazo.table :as t]))

(t/deftable photos
  "The photos table."
  (t/column :created-at :timestamp :not-null? true)
  (t/column :flickr-id :bigint :unique? true)
  (t/column :flickr-owner-id :text)
  (t/column :flickr-owner-name :text)
  (t/column :flickr-owner-url :text)
  (t/column :id :integer :primary-key? true)
  (t/column :location :geography :geometry :point)
  (t/column :status :text)
  (t/column :title :text)
  (t/column :likes :integer :not-null? true)
  (t/column :dislikes :integer :not-null? true)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :url :text)
  (t/column :user-id :integer))

(defn- select-expr [db]
  (record/select-columns db Photo))

(defn by-spot
  "Return all photos of `spot` from `db`."
  [db spot & [opts]]
  (->> @(sql/select db (select-expr db)
          (sql/from :photos)
          (sql/join :photos-spots.photo-id :photos.id)
          (sql/where `(= :photos-spots.spot-id ~(:id spot))))
       not-empty))

(s/fdef by-spot
  :args (s/cat :db sql/db?
               :spot :burningswell.db.spots/spot
               :opts (s/? (s/nilable map?)))
  :ret (s/nilable (s/coll-of ::photo)))

(defn dislike!
  "Dislike the given `photo` as `user`."
  [db user photo]
  (first @(sql/insert db :dislikes.photos []
            (sql/values [{:photo-id (:id photo)
                          :user-id (:id user)}])
            (sql/on-conflict [:photo-id :user-id]
              (sql/do-nothing))
            (sql/returning :*))))

(s/fdef dislike!
  :args (s/cat :db sql/db?
               :user :burningswell.db.users/user
               :photo :burningswell.db.photos/photo))

(defn like!
  "Like the given `photo` as `user`."
  [db user photo]
  (first @(sql/insert db :likes.photos []
            (sql/values [{:photo-id (:id photo)
                          :user-id (:id user)}])
            (sql/on-conflict [:photo-id :user-id]
              (sql/do-nothing))
            (sql/returning :*))))

(s/fdef like!
  :args (s/cat :db sql/db?
               :user :burningswell.db.users/user
               :photo :burningswell.db.photos/photo))

(defn unlike!
  "Unlike the given `photo` as `user`."
  [db user photo]
  (first @(sql/delete db :likes.photos
            (sql/where `(and (= :photo-id ~(:id photo))
                             (= :user-id ~(:id user))))
            (sql/returning :*))))

(s/fdef unlike!
  :args (s/cat :db sql/db?
               :user :burningswell.db.users/user
               :photo :burningswell.db.photos/photo))

(defn undislike!
  "Un-dislike the given `photo` as `user`."
  [db user photo]
  (first @(sql/delete db :dislikes.photos
            (sql/where `(and (= :photo-id ~(:id photo))
                             (= :user-id ~(:id user))))
            (sql/returning :*))))

(s/fdef undislike!
  :args (s/cat :db sql/db?
               :user :burningswell.db.users/user
               :photo :burningswell.db.photos/photo))

(defn search
  "Search photos."
  [db & [{:keys [direction distance limit location offset sort query] :as opts}]]
  @(sql/select db [:photos.id]
     (sql/from :photos)
     (util/fulltext query :title)
     (util/within-distance-to :location location distance {:spheroid? true})
     (sql/limit limit)
     (sql/offset offset)
     (if location
       (util/order-by-distance :location location)
       (util/order-by :photos opts))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::photo))

;; (defn- row [photo]
;;   (assoc (select-keys photo [:location :url :status :title])
;;          :flickr-id (-> photo :flickr :id)
;;          :flickr-owner-id (-> photo :flickr :owner :id)
;;          :flickr-owner-name (-> photo :flickr :owner :name)
;;          :flickr-owner-url (-> photo :flickr :owner :url)
;;          :user-id (-> photo :_embedded :user :id)))

;; (defn- select-all [db & [opts]]
;;   (select db [:photos.id
;;               (as '(cast :photos.location :geometry) :location)
;;               :photos.user-id
;;               :photos.flickr-id
;;               :photos.url
;;               :photos.status
;;               :photos.title
;;               :photos.likes
;;               :photos.dislikes
;;               :photos.created-at
;;               :photos.updated-at
;;               (as `(json_build_object
;;                     "id" :photos.flickr-id
;;                     "owner"
;;                     (case (and (is-null :photos.flickr-owner-id)
;;                                (is-null :photos.flickr-owner-name)
;;                                (is-null :photos.flickr-owner-url)) nil
;;                           (json_build_object
;;                            "id" :photos.flickr-owner-id
;;                            "name" :photos.flickr-owner-name
;;                            "url" :photos.flickr-owner-url)))
;;                   :flickr)]
;;     (from :photos)
;;     (fulltext (:query opts) :photos.title)
;;     (order-by (or (:order-by opts) (desc :photos.created-at)))))

;; (defn- select-details [db & [opts]]
;;   (select db [:photos.*
;;               :photo-likes.like
;;               (as `(json_build_object
;;                     "user" (json-embed-user :users))
;;                   :_embedded)]
;;     (from (as (select-all db opts) :photos))
;;     (join :users '(on (= :users.id :photos.user-id)) :type :left)
;;     (join :photo-likes
;;           `(on (and (= :photo-likes.photo-id :photos.id)
;;                     (= :photo-likes.user-id ~(-> opts :user :id))))
;;           :type :left)
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn all :- [Photo]
;;   "Return all photos in `db`."
;;   [db & [opts]]
;;   @(select-details db opts))

;; (s/defn insert-photos-spots
;;   [db :- Database photo :- Photo spot :- Spot & [user]]
;;   (->> @(sql/insert db :photos-spots []
;;           (values [{:photo-id (:id photo)
;;                     :spot-id (:id spot)
;;                     :user-id (:id user)}]))
;;        first :count pos?))

;; (s/defn update-photos-spots
;;   [db :- Database photo :- Photo spot :- Spot & [user]]
;;   (->> @(sql/update db :photos-spots
;;           {:photo-id (:id photo)
;;            :spot-id (:id spot)
;;            :user-id (:id user)}
;;           (where `(and (= :photos-spots.photo-id ~(:id photo))
;;                        (= :photos-spots.spot-id ~(:id spot)))))
;;        first :count pos?))

;; (s/defn add-spot
;;   "Associate `photo` with the `spot`."
;;   [db :- Database photo :- Photo spot :- Spot & [user]]
;;   (assert (:id spot))
;;   (assert (:id photo))
;;   (or (update-photos-spots db photo spot user)
;;       (insert-photos-spots db photo spot user)))

;; (s/defn by-id :- (s/maybe Photo)
;;   "Return the country in `db` by `id`."
;;   [db :- Database id :- s/Num & [opts]]
;;   (first @(compose
;;            (select-details db opts)
;;            (where `(= :photos.id (cast ~id :integer))))))

;; (defn by-flickr-id
;;   "Return the photo by it's Flickr `id`."
;;   [db id & [opts]]
;;   (first @(compose
;;            (select-details db opts)
;;            (where `(= :photos.flickr-id (cast ~id :bigint))))))

;; (defn assoc-images [db photos]
;;   (let [images (images/by-photos db photos)
;;         images (images/group-by-photo-id images)]
;;     (map (fn [photo]
;;            (->> (get images (:id photo))
;;                 (map images/select-embedded)
;;                 (images/zip-by-label)
;;                 (assoc photo :images)))
;;          photos)))

;; (s/defn by-countries
;;   "Return the cover photos of `countries` from `db`."
;;   [db :- Database countries :- [Country] & [opts]]
;;   (let [photo-ids (map (comp :id :photo :_embedded) countries)]
;;     (cond->> @(compose
;;                (select-details db opts)
;;                (where `(in :photos.id ~photo-ids)))
;;       (:images opts)
;;       (assoc-images db))))

;; (defn by-country
;;   "Return the photos of `country` from `db`."
;;   [db country & [opts]]
;;   (->> @(compose
;;          (select-details db opts)
;;          (join :photos-countries.photo-id :photos.id)
;;          (where `(= :photos-countries.country-id ~(:id country))))
;;        (:images opts)
;;        (assoc-images db)))

;; (s/defn by-spots
;;   "Return the cover photos of `spots` from `db`."
;;   [db :- Database spots :- [Spot] & [opts]]
;;   (let [photo-ids (map (comp :id :photo :_embedded) spots)]
;;     (cond->> @(compose
;;                (select-details db opts)
;;                (where `(in :photos.id ~photo-ids)))
;;       (:images opts)
;;       (assoc-images db))))

;; (s/defn by-regions
;;   "Return the cover photos of `regions` from `db`."
;;   [db :- Database regions :- [Region] & [opts]]
;;   (let [photo-ids (map (comp :id :photo :_embedded) regions)]
;;     (cond->> @(compose
;;                (select-details db opts)
;;                (where `(in :photos.id ~photo-ids)))
;;       (:images opts)
;;       (assoc-images db))))

;; (defn by-region
;;   "Return the photos of `region` from `db`."
;;   [db region & [opts]]
;;   (cond->> @(compose
;;              (select-details db opts)
;;              (join :photos-regions.photo-id :photos.id)
;;              (where `(= :photos-regions.region-id ~(:id region))))
;;     (:images opts)
;;     (assoc-images db)))

;; (s/defn delete
;;   "Delete `photo` from `db`."
;;   [db :- Database photo :- Photo]
;;   (->> @(sql/delete db :photos
;;           (where `(= :photos.id
;;                      ~(:id photo))))
;;        first :count))

;; (defn delete-by-spot
;;   "Delete all the photos of `spot` in `db`."
;;   [db spot]
;;   @(sql/delete db :photos
;;      (sql/where `(in :id ~(sql/select db [:photo-id]
;;                             (sql/from :photos-spots)
;;                             (where `(= :photos-spots.spot-id ~(:id spot))))))))

;; (s/defn insert
;;   "Insert `photo` into `db`."
;;   [db :- Database photo]
;;   (->> @(sql/insert db :photos []
;;           (values [(row photo)])
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn update
;;   "Update `photo` in `db`."
;;   [db :- Database photo]
;;   (->> @(sql/update db :photos
;;           (row photo)
;;           (where `(or (= :photos.id ~(:id photo))
;;                       (= :photos.flickr-id ~(-> photo :flickr :id))))
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn save :- Photo
;;   "Save `photo` to `db`."
;;   [db :- Database photo]
;;   (or (update db photo)
;;       (insert db photo)))

;; (s/defn with-images :- [Photo]
;;   "Return all photos in `db` with their images."
;;   [db :- Database & [opts]]
;;   (let [photos (all db opts)]
;;     (assoc-images db photos)))

;; (s/defn with-images-by-id :- Photo
;;   "Return the photo in `db` by `id` with it's images."
;;   [db :- Database id :- s/Num & [opts]]
;;   (if-let [photo (by-id db id opts)]
;;     (first (assoc-images db [photo]))))
