(ns burningswell.db.addresses
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.select-batch :as select-batch]
            [datumbazo.table :as t]))

(t/deftable addresses
  "The addresses table."
  (t/column :city :citext)
  (t/column :country-id :integer)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :formatted :citext :not-null? true)
  (t/column :id :integer :primary-key? true :not-null? true)
  (t/column :location :geography :geometry :point :srid 4326
            :not-null? true :unique? true)
  (t/column :postal-code :citext)
  (t/column :region-id :integer)
  (t/column :street-name :citext)
  (t/column :street-number :citext)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :user-id :integer))

(defn ke-nui-road-haleiwa
  "Return the address 59-343 Ke Nui Road, Haleiwa, HI 96712, USA."
  [db]
  (by-id db 7))

(defn by-locations
  "Resolve addresses by their `locations`."
  [db locations]
  (let [locations (map (partial hash-map :location) locations)]
    (select-batch/select-batch db :addresses locations {:join {:location :geometry}})))

(defn search
  "Search addresses."
  [db & [{:keys [countries direction distance except limit location
                 offset regions sort query] :as opts}]]
  @(sql/select db [:addresses.id]
               (sql/from :addresses)
               (util/fulltext query :addresses.name)
               (util/within-distance-to :location location distance {:spheroid? true})
               (when-not (empty? countries)
                 (sql/where `(in :country-id ~(seq countries)) :and))
               (when-not (empty? except)
                 (sql/where `(not (in :id ~(seq except))) :and))
               (when-not (empty? regions)
                 (sql/where `(in :region-id ~(seq regions)) :and))
               (some-> limit sql/limit)
               (some-> offset sql/offset)
               (if location
                 (util/order-by-distance :location location)
                 (util/order-by :addresses opts))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::address))

;; (defn- select-all
;;   "Returns all addresses."
;;   [db & [opts]]
;;   (let [{:keys [query page per-page]} opts]
;;     (select db [:addresses.id
;;                 :addresses.formatted
;;                 :addresses.street-name
;;                 :addresses.street-number
;;                 :addresses.postal-code
;;                 :addresses.city
;;                 (as '(cast :addresses.location :geometry) :location)
;;                 :addresses.created-at
;;                 :addresses.updated-at
;;                 (as `(json_build_object
;;                       "country" (json-embed-country :countries)
;;                       "region" (json-embed-region :regions)
;;                       "user" (json-embed-user :users))
;;                     :_embedded)]
;;       (from :addresses)
;;       (join :countries.id :addresses.country-id)
;;       (join :regions.id :addresses.region-id :type :left)
;;       (join :users.id :addresses.user-id :type :left)
;;       (fulltext query :addresses.formatted)
;;       (cond
;;         (:location opts)
;;         (order-by-distance :addresses.location (:location opts))
;;         :else (order-by :addresses.formatted))
;;       (paginate page per-page))))

;; (defn- row [spot]
;;   (assoc (select-keys spot [:formatted :city :location :street-name
;;                             :street-number :postal-code])
;;          :country-id (-> spot :_embedded :country :id)
;;          :region-id (-> spot :_embedded :region :id)
;;          :user-id (-> spot :_embedded :user :id)))

;; (s/defn all :- [Address]
;;   "Return all addresses in `db`."
;;   [db :- Database & [opts]]
;;   @(select-all db opts))

;; (s/defn by-id :- (s/maybe Address)
;;   "Return the country in `db` by `id`."
;;   [db :- Database id :- s/Num]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :addresses.id (cast ~id :integer))))))

;; (s/defn by-formatted :- (s/maybe Address)
;;   "Return the country in `db` by `formatted`."
;;   [db :- Database formatted :- s/Str]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :addresses.formatted ~formatted)))))

;; (s/defn by-location :- (s/maybe Address)
;;   "Return the country in `db` by `location`."
;;   [db :- Database location :- Point]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :addresses.location ~location)))))

;; (s/defn delete
;;   "Delete `address` from `db`."
;;   [db :- Database address :- Address]
;;   (->> @(sql/delete db :addresses
;;           (where `(= :addresses.id
;;                      ~(:id address))))
;;        first :count))

;; (s/defn insert :- Address
;;   "Insert `address` into `db`."
;;   [db :- Database address]
;;   (->> @(sql/insert db :addresses []
;;           (values [(row address)])
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn update :- Address
;;   "Update `address` in `db`."
;;   [db :- Database address]
;;   (->> @(sql/update db :addresses
;;           (row address)
;;           (where `(or (= :addresses.id (cast ~(:id address) :integer))
;;                       (= :addresses.location ~(:location address))))
;;           (returning :id))
;;        first :id (by-id db)))
