(ns burningswell.db.airports
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]))

(t/deftable airports
  "The airport table."
  (t/column :country-id :integer)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :gps-code :citext)
  (t/column :iata-code :citext)
  (t/column :id :integer :primary-key? true)
  (t/column :location :geography)
  (t/column :name :citext)
  (t/column :region-id :integer)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :wikipedia-url :text))

(defn honolulu
  "Returns the Honolulu airport."
  [db]
  (first (by-iata-code db "HNL")))

(defn search
  "Search airports."
  [db {:keys [countries direction distance except
              limit location offset regions sort query] :as opts}]
  @(sql/select db [:airports.id]
     (sql/from :airports)
     (util/fulltext query :name)
     (util/within-distance-to :location location distance {:spheroid? true})
     (when-not (empty? countries)
       (sql/where `(in :country-id ~(seq countries)) :and))
     (when-not (empty? except)
       (sql/where `(not (in :id ~(seq except))) :and))
     (when-not (empty? regions)
       (sql/where `(in :region-id ~(seq regions)) :and))
     (sql/limit limit)
     (sql/offset offset)
     (if location
       (util/order-by-distance :location location)
       (util/order-by :airports opts))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? (s/nilable map?)))
  :ret (s/coll-of ::airport))
