(ns burningswell.db.images
  (:require [burningswell.db.util :as util]
            [clojure.string :as str]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]
            [clojure.spec.alpha :as s]))

(t/deftable images
  "The images table."
  (t/column :content-disposition :text)
  (t/column :content-length :integer)
  (t/column :content-md5 :text)
  (t/column :content-type :text)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :height :integer)
  (t/column :id :integer :primary-key? true)
  (t/column :label :text)
  (t/column :photo-id :integer)
  (t/column :storage-key :text)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :url :text)
  (t/column :width :integer))

(defn by-photo
  "Return the images of `photo` from `db`."
  [db photo & [opts]]
  @(sql/select db [:*]
     (sql/from :images)
     (sql/where `(= :images.photo-id ~(:id photo)))
     (sql/order-by :images.width)))

(s/fdef by-photo
  :args (s/cat :db sql/db?
               :photo :burningswell.db.photos/photo
               :opts (s/? (s/nilable map?)))
  :ret (s/coll-of ::image))

(defn by-photo-and-label
  "Return the images of `photo` from `db`."
  [db photo label & [opts]]
  (first @(sql/select db [:*]
            (sql/from :images)
            (sql/where `(and (= :images.photo-id ~(:id photo))
                             (= :images.label ~label))))))

(s/fdef by-photo-and-label
  :args (s/cat :db sql/db?
               :photo :burningswell.db.photos/photo
               :label ::label
               :opts (s/? (s/nilable map?)))
  :ret (s/nilable ::image))

(defn by-photos
  "Return the images of `photos` from `db`."
  [db photos & [opts]]
  (->> {:order-by '(order-by :images.width)}
       (sql/has-many db photos :photos :images)))

(s/fdef by-photos
  :args (s/cat :db sql/db?
               :photo (s/coll-of :burningswell.db.photos/photo)
               :opts (s/? (s/nilable map?))))

(defn search
  "Search images."
  [db & [{:keys [direction limit offset photos sort query] :as opts}]]
  @(sql/select db [:images.id]
     (sql/from :images)
     (util/fulltext query :address)
     (when-not (empty? photos)
       (sql/where `(in :photo-id ~(seq photos)) :and))
     (sql/limit limit)
     (sql/offset offset)
     (util/order-by :images opts)))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::image))

;; (def embedded-keys
;;   "The keys for an embedded image."
;;   [:id :content-length :content-type :height :label :url
;;    :width :created-at :updated-at])

;; (defn- row [image]
;;   (assoc (select-keys image [:content-disposition :content-length :content-md5
;;                              :content-type :width :height :label :storage-key
;;                              :url])
;;          :photo-id (-> image :_embedded :photo :id)))

;; (defn- select-all [db & [opts]]
;;   (select db [:images.id
;;               :images.content-disposition
;;               :images.content-length
;;               :images.content-md5
;;               :images.content-type
;;               :images.width
;;               :images.height
;;               :images.label
;;               :images.storage-key
;;               :images.url
;;               :images.created-at
;;               :images.updated-at
;;               (as `(json_build_object
;;                     "photo"
;;                     (json_build_object
;;                      "id" :images.photo-id))
;;                   :_embedded)]
;;     (from :images)
;;     (order-by (or (:order-by opts)
;;                   (desc :images.width)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn all :- [Image]
;;   "Return all images in `db`."
;;   [db :- Database & [opts]]
;;   @(select-all db opts))

;; (s/defn by-id :- (s/maybe Image)
;;   "Return the country in `db` by `id`."
;;   [db :- Database id :- s/Num]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :images.id (cast ~id :integer))))))

;; (s/defn by-photos
;;   "Return the images of `photos` from `db`."
;;   [db :- Database photos :- [Photo] & [opts]]
;;   @(compose
;;     (select-all db opts)
;;     (where `(in :images.photo-id ~(map :id photos)))))

;; (s/defn delete
;;   "Delete `image` from `db`."
;;   [db :- Database image :- Image]
;;   (->> @(sql/delete db :images
;;           (where `(= :images.id
;;                      ~(:id image))))
;;        first :count))

;; (s/defn insert
;;   "Insert `image` into `db`."
;;   [db :- Database image]
;;   (->> @(sql/insert db :images []
;;           (values [(row image)])
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn update
;;   "Update `image` in `db`."
;;   [db :- Database image]
;;   (some->> @(sql/update db :images
;;               (row image)
;;               (where `(or (= :images.id ~(:id image))
;;                           (and (= :images.photo-id
;;                                   ~(-> image :_embedded :photo :id))
;;                                (= :images.label ~(:label image)))))
;;               (returning :id))
;;            first :id (by-id db)))

;; (s/defn save
;;   "Save `image` to `db`."
;;   [db :- Database image]
;;   (or (update db image)
;;       (insert db image)))

;; (defn select-embedded
;;   "Return an embedded version of `image`."
;;   [image]
;;   (select-keys image embedded-keys))

;; (defn photo-id
;;   "Return the photo-id of `image`."
;;   [image]
;;   (get-in image [:_embedded :photo :id]))

;; (defn group-by-photo-id
;;   "Group `images` by their photo."
;;   [images]
;;   (core/group-by photo-id images))

;; (defn zip-by-label
;;   "Zip the `images` by their label."
;;   [images]
;;   (zipmap (map (comp keyword :label) images) images))
