(ns burningswell.db.continents
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]))

(t/deftable continents
  "The continents table."
  (t/column :code :citext :not-null? true :unique? true)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :geom :geography)
  (t/column :id :serial :primary-key? true)
  (t/column :location :geography)
  (t/column :name :citext :not-null? true :unique? true)
  (t/column :updated-at :timestamp :not-null? true))

(defn asia
  "Find the continent Asia."
  [db]
  (by-name db "Asia"))

(s/fdef asia
  :args (s/cat :db sql/db?))

(defn europe
  "Find the continent Europe."
  [db]
  (by-name db "Europe"))

(s/fdef europe
  :args (s/cat :db sql/db?))

(defn north-america
  "Find the the continent North America."
  [db]
  (by-name db "North America"))

(s/fdef north-america
  :args (s/cat :db sql/db?))

(defn search
  "Search continents."
  [db & [{:keys [after before distance first last location sort query]}]]
  @(sql/select db [:continents.id]
     (sql/from :continents)
     (util/fulltext query :name)
     (util/within-distance-to :geom location distance {:spheroid? true})
     (sql/limit first)
     (sql/offset (or (:offset after) 0))
     (if location
       (util/order-by-distance :location location)
       (sql/order-by (keyword (or sort :name))))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::continent))

(defn select-top-photos
  "Return the top photo for each continent."
  [db]
  (sql/select db (sql/distinct
                  [:photos-continents.continent-id
                   :photos-continents.photo-id]
                  :on [:photos-continents.continent-id] )
    (sql/from :photos-continents)
    (sql/join :photos.id :photos-continents.photo-id)
    (sql/order-by :photos-continents.continent-id
                  (sql/desc :photos.created-at))))

(s/fdef select-top-photos
  :args (s/cat :db sql/db?))

(defn update-photos!
  "Update the photos of all continents."
  [db]
  (first @(sql/update db :continents
            {:photo-id :top-photos.photo-id}
            (sql/from (sql/as (select-top-photos db) :top-photos))
            (sql/where `(= :continents.id :top-photos.continent-id)))))

(s/fdef update-photos!
  :args (s/cat :db sql/db?))

;; (defn- select-all [db & [opts]]
;;   (select db [:continents.id
;;               :continents.name
;;               :continents.code
;;               :continents.country-count
;;               :continents.created-at
;;               :continents.updated-at]
;;     (from :continents)
;;     (fulltext (:query opts) :continents.name)
;;     (order-by-distance :continents.geom (:location opts))
;;     (order-by :continents.name)
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn all :- [Continent]
;;   "Return all continents in `db`."
;;   [db :- Database & [opts]]
;;   @(select-all db opts))

;; (s/defn by-code :- (s/maybe Continent)
;;   "Return the continent in `db` by `code`."
;;   [db :- Database code :- s/Str]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :continents.code
;;                       ~(str/lower-case code))))))

;; (s/defn by-id :- (s/maybe Continent)
;;   "Return the continent in `db` by `id`."
;;   [db :- Database id :- s/Int]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :continents.id (cast ~id :integer))))))

;; (s/defn by-name :- (s/maybe Continent)
;;   "Return the continent in `db` by `name`."
;;   [db :- Database name :- s/Str]
;;   (first @(compose
;;            (select-all db)
;;            (where `(= :continents.name ~name)))))

;; (s/defn by-country :- (s/maybe Continent)
;;   "Load the region of `country` from `db`."
;;   [db :- Database country :- Country]
;;   (by-id db (-> country :_embedded :continent :id)))

;; (s/defn by-spot :- (s/maybe Continent)
;;   "Return the continent in `db` by `name`."
;;   [db :- Database spot :- Spot]
;;   (let [country-id (-> spot :_embedded :country :id)]
;;     (first @(compose
;;              (select-all db)
;;              (join :countries.continent-id :continents.id)
;;              (where `(= :countries.id ~country-id))))))

;; (defn- row [continent]
;;   (select-keys continent [:name :code]))

;; (s/defn delete
;;   "Delete `continent` from `db`."
;;   [db :- Database continent :- Continent]
;;   (->> @(sql/delete db :continents
;;           (where `(= :continents.id
;;                      ~(:id continent))))
;;        first :count))

;; (s/defn insert :- Continent
;;   "Insert `continent` into `db`."
;;   [db :- Database continent]
;;   (->> @(sql/insert db :continents []
;;           (values [(row continent)])
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn update :- Continent
;;   "Update `continent` in `db`."
;;   [db :- Database continent]
;;   (->> @(sql/update db :continents (row continent)
;;                     (where `(= :continents.id
;;                                ~(:id continent)))
;;                     (returning :id))
;;        first :id (by-id db)))

;; (s/defn update-counters
;;   "Update the counters of all continents."
;;   [db :- Database]
;;   (->> [(select db ['(update-continent-country-count)])]
;;        (map (comp first deref))
;;        (apply merge)))
