(ns burningswell.db.continents
  (:require [burningswell.db.page-views :as page-views]
            [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]
            [geo.postgis :as geo]))

(t/deftable continents
  "The continents table."
  (t/column :code :citext :not-null? true :unique? true)
  (t/column :country-count :integer :default 0 :not-null? true)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :geom :geography)
  (t/column :id :serial :primary-key? true)
  (t/column :location :geography)
  (t/column :name :citext :not-null? true :unique? true)
  (t/column :region-count :integer :default 0 :not-null? true)
  (t/column :spot-count :integer :default 0 :not-null? true)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :user-count :integer :default 0 :not-null? true))

(defn asia
  "Find the continent Asia."
  [db]
  (by-name db "Asia"))

(s/fdef asia
  :args (s/cat :db sql/db?))

(defn europe
  "Find the continent Europe."
  [db]
  (by-name db "Europe"))

(s/fdef europe
  :args (s/cat :db sql/db?))

(defn north-america
  "Find the the continent North America."
  [db]
  (by-name db "North America"))

(s/fdef north-america
  :args (s/cat :db sql/db?))

(defn page-views
  "Returns the page views for `continent`."
  [db continent & [opts]]
  (first (page-views/totals db :continents [continent] opts)))

(s/fdef page-views
  :args (s/cat :db sql/db? :continent ::continent :opts (s/? (s/nilable map?))))

(defn search
  "Search continents."
  [db & [{:keys [distance direction except limit location
                 offset sort query] :as opts}]]
  @(sql/select db [:public.continents.id]
     (sql/from :continents)
     (util/fulltext query :name)
     (util/within-distance-to :geom location distance {:spheroid? true})
     (when (= sort :views)
       (sql/join :page-views.continents.continent-id :public.continents.id :type :left))
     (when-not (empty? except)
       (sql/where `(not (in :id ~(seq except))) :and))
     (sql/limit limit)
     (sql/offset offset)
     (cond
       (= sort :views)
       (util/order-by-views :continents opts)
       (and location (= sort :distance))
       (util/order-by-distance :location location opts)
       sort
       (util/order-by :continents opts)
       :else (sql/order-by :public.continents.name))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::continent))

(defn select-top-photos
  "Return the top photo for each continent."
  [db]
  (sql/select db (sql/distinct
                  [:photos-continents.continent-id
                   :photos-continents.photo-id]
                  :on [:photos-continents.continent-id] )
    (sql/from :photos-continents)
    (sql/join :photos.id :photos-continents.photo-id)
    (sql/order-by :photos-continents.continent-id
                  (sql/desc :photos.created-at))))

(s/fdef select-top-photos
  :args (s/cat :db sql/db?))

(defn select-infer-centroids
  "Infer the centroids of all continents from the :geom column in
  the :countries table."
  [db]
  (sql/select db [(sql/as :continent-id :id)
                  (sql/as '(st_centroid
                            (st_union
                             (st_centroid
                              (cast :geom :geometry))))
                          :location)]
    (sql/from :countries)
    (sql/group-by :continent-id)))

(s/fdef select-infer-centroids
  :args (s/cat :db sql/db?))

(defn update-photos!
  "Update the photos of all continents."
  [db]
  (first @(sql/update db :continents
            {:photo-id :top-photos.photo-id}
            (sql/from (sql/as (select-top-photos db) :top-photos))
            (sql/where `(= :continents.id :top-photos.continent-id)))))

(s/fdef update-photos!
  :args (s/cat :db sql/db?))

(defn update-counters!
  "Update the counters of all continents."
  [db]
  (->> [(sql/select db ['(update-continent-country-count)
                        '(update-continent-region-count)
                        '(update-continent-spot-count)
                        '(update-continent-user-count)])]
       (map (comp first deref))
       (apply merge)))

(s/fdef update-counters!
  :args (s/cat :db sql/db?))

(defn update-centroids!
  "Infer and update the centroids of all continents from the :geom
  column in the :countries table."
  [db]
  @(sql/update db :continents
     {:location :u.location}
     (sql/where '(= :continents.id :u.id))
     (sql/from (sql/as (select-infer-centroids db) :u))
     (sql/returning :*)))
