(ns burningswell.db.favourites
  (:require [burningswell.db.continents :as continents]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.select-batch :as batch]
            [inflections.core :as infl]
            [medley.core :refer [map-vals]]))

(s/def ::favourite map?)

(defn- cast-type [type]
  `(cast ~(name type) :favourite_type))

(defn table
  "Returns the table for `type`."
  [type]
  (keyword (str "favourites." (infl/plural (name type)))))

(defn foreign-key
  "Returns the table for `type`."
  [type]
  (keyword (str (name type) "-id")))

(defn author?
  "Returns true of `user` is the author of `favourite`, otherwise false."
  [db favourite user]
  (->> @(sql/select db [1]
          (sql/from :favourites)
          (sql/where `(and (= :id ~(:id favourite))
                           (= :author-id ~(:id user)))))
       empty? not))

(s/fdef author?
  :args (s/cat :db sql/db?
               :favourite ::favourite
               :user (s/nilable :burningswell.db.users/user)))

(defn create!
  "Insert the `favourite` into the `db`."
  [db favourite]
  (first @(sql/insert db (table (:type favourite)) []
            (sql/values [(update favourite :type cast-type)])
            (sql/returning :*))))

(s/fdef create!
  :args (s/cat :db sql/db? :favourite ::favourite))

(defn create-continent!
  "Create a continent favourite."
  [db author continent]
  (create! db {:author-id (:id author)
               :continent-id (:id continent)
               :type :continent}))

(s/fdef create-continent!
  :args (s/cat :db sql/db?
               :author :burningswell.db.users/user
               :continent :burningswell.db.continents/continent))

(defn create-country!
  "Create a country favourite."
  [db author country]
  (create! db {:author-id (:id author)
               :country-id (:id country)
               :type :country}))

(s/fdef create-country!
  :args (s/cat :db sql/db?
               :author :burningswell.db.users/user
               :country :burningswell.db.countries/country))

(defn create-region!
  "Create a region favourite."
  [db author region]
  (create! db {:author-id (:id author)
               :region-id (:id region)
               :type :region}))

(s/fdef create-region!
  :args (s/cat :db sql/db?
               :author :burningswell.db.users/user
               :region :burningswell.db.regions/region))

(defn create-spot!
  "Create a spot favourite."
  [db author spot]
  (create! db {:author-id (:id author)
               :spot-id (:id spot)
               :type :spot}))

(s/fdef create-region!
  :args (s/cat :db sql/db?
               :author :burningswell.db.users/user
               :spot :burningswell.db.spots/spot))

(defn delete!
  "Delete a favourite from `db`."
  [db favourite]
  (first @(sql/delete db :favourites
            (sql/where `(= :id ~(:id favourite)))
            (sql/returning :*))))

(s/fdef delete!
  :args (s/cat :db sql/db? :favourite ::favourite))

(defn resolve-batch
  "Resolve a batch of favourites."
  ([db favourites]
   (batch/select-batch db :favourites favourites))
  ([db table favourites]
   (batch/select-batch db table favourites)))

(defn favourite?
  "Returns true if the favourite for `obj` and `type` exists, otherwise false."
  [db author type {:keys [id]}]
  (->> @(sql/select db [1]
          (sql/from (table type))
          (sql/where `(and (= :author-id ~(:id author))
                           (= ~(foreign-key type) ~id))))
       empty? not))

(defn resolve-favourite
  [db author type entities]
  (let [foreign-key (foreign-key type)
        result @(sql/select db [:*]
                  (sql/from (table type))
                  (sql/where `(and (= :author-id ~(:id author))
                                   (in ~foreign-key ~(map :id entities)))))]
    (map (map-vals first (group-by foreign-key result))
         (map :id entities))))

(defn- cast-favourite-types [types]
  (map (fn [type] `(cast ~(name type) :favourite_type)) types))

(defn resolve-user-favourites
  [db users & [opts]]
  (->> {:limit (:limit opts)
        :offset (:offset opts)
        :order-by (or (:order-by opts)
                      `(~'order-by
                        (~(or (:direction opts) 'desc)
                         :favourites.updated-at)))
        :source-fk :favourites.author-id
        :where (when-let [types (:types opts)]
                 (sql/where `(in :type ~(cast-favourite-types types))))}
       (sql/has-many db users :users :favourites)))

(defn by-id
  "Returns the favourite by `id`."
  ([db id]
   (when-let [favourite (by-id db :favourites id)]
     (by-id db (table (:type favourite)) id)))
  ([db table id]
   (first @(sql/select db [:*]
             (sql/from table)
             (sql/where `(= :id ~id))))))

(defn target
  "Returns the target of the favourite."
  [db {:keys [id type] :as favourite}]
  (case (keyword type)
    :continent (continents/by-id db (:continent-id favourite))
    :country (countries/by-id db (:country-id favourite))
    :region (regions/by-id db (:region-id favourite))
    :spot (spots/by-id db (:spot-id favourite))))
