(ns burningswell.kafka.cluster
  (:require [burningswell.kafka.core :as core]
            [burningswell.zookeeper.cluster :as zookeeper]
            [peripheral.core :refer [defcomponent]])
  (:import kafka.server.KafkaServerStartable))

(defn config
  "Returns the default Kafka cluster config."
  [& [opts]]
  (let [bind-port (core/random-port)]
    (merge {:bind-port bind-port
            :bootstrap.servers (str "localhost:" bind-port)
            :controller.socket.timeout.ms 1000
            :listeners (str "PLAINTEXT://localhost:" bind-port)
            :log.dirs (core/random-filename "kafka" "log")
            :log.flush.interval.messages 1
            :log.flush.scheduler.interval.ms 100
            :offsets.topic.replication.factor 1
            :replica.socket.timeout.ms 1000}
           opts)))

(defn- cleanup
  "Cleanup the Kafka cluster."
  [{:keys [config] :as cluster}]
  (when (:delete? config)
    (-> config :log.dirs core/delete-dir)))

(defn- start-server
  "Start the Kafka server."
  [{:keys [config zookeeper] :as kafka}]
  (doto (KafkaServerStartable/fromProps
         (->> (zookeeper/connection zookeeper)
              (assoc config :zookeeper.connect)
              (core/properties)))
    (.startup)))

(defn- shutdown
  "Shutdown the Kafka server."
  [server]
  (.shutdown server)
  ;; (.awaitShutdown server)
  )

(defcomponent Cluster [config zookeeper]
  :this/as *this*
  :server (start-server *this*) #(shutdown %)
  :on/stopped (cleanup *this*))

(defn cluster
  "Returns an embedded Kafka cluster."
  [& [opts]]
  (map->Cluster {:config (config opts)}))
