(ns burningswell.kafka.core
  (:import org.apache.kafka.clients.consumer.ConsumerConfig
           org.apache.commons.io.FileUtils
           org.apache.kafka.streams.StreamsConfig
           rads.kafka.transit.MsgpackSerde)
  (:require [clojure.java.io :as io]))

(def bootstrap-servers
  "The Kafka default bootstrap servers."
  "localhost:9092")

(defn delete-dir
  "Delete the `directory`."
  [directory]
  (FileUtils/deleteDirectory (io/file directory)))

(defn props [application-id]
  {ConsumerConfig/AUTO_OFFSET_RESET_CONFIG "earliest"
   StreamsConfig/APPLICATION_ID_CONFIG application-id
   StreamsConfig/BOOTSTRAP_SERVERS_CONFIG bootstrap-servers
   StreamsConfig/CACHE_MAX_BYTES_BUFFERING_CONFIG 0
   StreamsConfig/DEFAULT_KEY_SERDE_CLASS_CONFIG (.getName MsgpackSerde)
   StreamsConfig/DEFAULT_VALUE_SERDE_CLASS_CONFIG (.getName MsgpackSerde)})

(defn properties
  "Convert the map `m` into a java.util.Properties object."
  [m]
  (let [props (java.util.Properties.)]
    (doseq [[k v] m] (.setProperty props (name k) (str v)))
    props))

(defn random-port
  "Returns a random free port number."
  []
  (with-open [socket (java.net.ServerSocket. 0)]
    (.getLocalPort socket)))

(defn random-filename
  "Returns a random filename."
  [& parts]
  (str (io/file (apply io/file (System/getProperty "java.io.tmpdir") parts)
                (str (java.util.UUID/randomUUID)))))

(defn config []
  (let [bootstrap-port (random-port)
        zookeeper-port (random-port)]
    {:bootstrap.servers (str "localhost:" bootstrap-port)
     :zookeeper.connect (str "localhost:" zookeeper-port)}))

(defn node->map
  "Convert a node object to a map."
  [node]
  {:empty? (.isEmpty node)
   :host (.host node)
   :id (.id node)
   :port (.port node)
   :rack (.rack node)
   :rack? (.hasRack node)})

(defn partition-info->map
  "Convert a partition info object to a map."
  [part-info]
  {:in-sync-replicas (map node->map (.inSyncReplicas part-info))
   :leader (node->map (.leader part-info))
   :partition (.partition part-info)
   :replicas (map node->map (.replicas part-info))
   :topic (.topic part-info)})
