(ns burningswell.zookeeper.cluster
  (:require [burningswell.kafka.core :as core]
            [clojure.java.io :as io]
            [peripheral.core :refer [defcomponent]])
  (:import java.net.InetSocketAddress
           [org.apache.zookeeper.server NIOServerCnxnFactory ZooKeeperServer]))

(defn config
  "Returns the default Zookeeper cluster config."
  [& [opts]]
  (merge {:bind-address "localhost"
          :bind-port (core/random-port)
          :log-dir (core/random-filename "zookeeper" "log")
          :snapshot-dir (core/random-filename "zookeeper" "snapshot")
          :tick-time 500}
         opts))

(defn- cleanup
  "Cleanup the Zookeeper cluster."
  [{:keys [config] :as cluster}]
  (when (:delete? config)
    (-> config :snapshot-dir core/delete-dir)
    (-> config :log-dir core/delete-dir)))

(defn- make-server
  "Make a Zookeeper server from `config`."
  [{:keys [config] :as cluster}]
  (ZooKeeperServer.
   (-> config :snapshot-dir io/file)
   (-> config :log-dir io/file)
   (:tick-time config)))

(defn- start-factory
  "Start the Zookeeper factory."
  [{:keys [config server] :as cluster}]
  (doto (NIOServerCnxnFactory.)
    (.configure (InetSocketAddress.
                 (:bind-address config)
                 (:bind-port config))
                60)
    (.startup server)))

(defn- shutdown-factory
  "Shutdown the Zookeeper factory."
  [factory]
  (.shutdown factory)
  ;; (.join factory)
  )

(defcomponent Cluster [config]
  :this/as *this*
  :server (make-server *this*)
  :factory (start-factory *this*) #(shutdown-factory %)
  ;; TODO: Needed?
  :wait (while (not (.isRunning server)) (Thread/sleep 100))
  :on/stopped (cleanup *this*))

(defn cluster
  "Returns an embedded Zookeeper cluster."
  [& [opts]]
  (map->Cluster {:config (config opts)}))

(defn connection
  "Returns the Zookeeper connection string."
  [{:keys [config] :as zookeeper}]
  (str (:bind-address config) ":" (:bind-port config)))
