(ns burningswell.metrics.core
  (:require [iapetos.core :as prometheus]
            [iapetos.export :as export]
            [iapetos.collector.jvm :as jvm]
            [io.pedestal.interceptor :refer [interceptor]])
  (:import iapetos.registry.Registry))

(def http-requests-total
  "The total HTTP request counter."
  (prometheus/counter
   :http/requests-total
   {:description "Number of requests."
    :labels [:method :uri :status :route]}))

(def http-request-duration-seconds
  "The HTTP request duration histogram."
  (prometheus/histogram
   :http/request-duration-seconds
   {:description "Request duration by url."
    :labels [:method :uri :status :route]}))

(defn metrics
  "Returns a new metrics component."
  [& [opts]]
  (-> (prometheus/collector-registry)
      (jvm/initialize)
      (prometheus/register http-requests-total)
      (prometheus/register http-request-duration-seconds)))

(defn export
  "Return the API metrics."
  [{:keys [metrics]}]
  {:status 200
   :headers {"Content-Type" "text/plain"}
   :body (export/text-format metrics)})

(defn- calculate-duration-seconds
  "Calculate the request duration in milliseconds
  from ::request-started-at in `context` and the current time."
  ([context]
   (calculate-duration-seconds context (System/currentTimeMillis)))
  ([context now]
   (when-let [startet-at (::request-started-at context)]
     (/ (- now startet-at) 1000.0))))

(defn- request-labels
  "Return the request measurement labels."
  [context]
  {:method (-> context :request :request-method name)
   :route (some-> context :route :route-name name)
   :status (-> context :response :status)
   :uri (-> context :request :uri)})

(defn- request-duration-seconds-request
  [context]
  (assoc context ::request-started-at (System/currentTimeMillis)))

(defn- request-duration-seconds-response
  "Calculate the request duration in milliseconds
  from ::request-started-at in `context` and the current time."
  [context]
  (prometheus/observe
   (-> context :request :metrics)
   :http/request-duration-seconds
   (request-labels context)
   (calculate-duration-seconds context))
  context)

(defn- requests-total-response
  "Calculate the request duration in milliseconds
  from ::request-started-at in `context` and the current time."
  [context]
  (prometheus/inc
   (-> context :request :metrics)
   :http/requests-total
   (request-labels context))
  context)

(defn metrics-interceptor
  "Return the metrics interceptor."
  [& [opts]]
  (interceptor
   {:name ::metrics
    :enter
    (fn [context]
      (request-duration-seconds-request context))
    :leave
    (fn [context]
      (-> (request-duration-seconds-response context)
          (requests-total-response)))}))
