(ns burningswell.streams.spots.flickr-photos
  (:require [burningswell.db.countries :as countries]
            [burningswell.services.photos :as photo-service]
            [burningswell.streams.api.events :as events]
            [burningswell.streams.core :as k]
            [peripheral.core :refer [defcomponent]]
            [burningswell.db.photo-labels :as photo-labels]
            [burningswell.db.photos :as photos]
            [burningswell.db.spots :as spots]
            [burningswell.services.flickr :as flickr]
            [burningswell.io :refer [slurp-byte-array]]
            [burningswell.services.vision :as vision]
            [clj-http.client :as http]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [geo.core :as geo]
            [geo.postgis :refer [point]]
            [medley.core :refer [distinct-by]]
            [sqlingvo.core :as sql]))

(def commercial-licenses
  "Flickr licenses that allow photos to be used on commercial sites."
  (map :id (filter :usable? flickr/licenses)))

(def include-labels
  "Photos with these labels are included."
  #{"coast" "ocean" "surf" "surfing" "wind wave"})

(def exclude-labels
  "Photos with these labels are excluded."
  #{"man" "woman" "vacation"})

(defn config [& [opts]]
  (->> {:application.id "spot-photos"
        :input {:events "burningswell.api.events"}
        :output {:created "burningswell.spots.photos.created"}
        :num.stream.threads 2}
       (merge opts)))

(defn search-criterias
  "Return a vector of search criterias for `spot`."
  [country {:keys [location] :as spot}]
  (let [license (str/join ","  commercial-licenses)]
    [{:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str (:name spot) ", " (:name country))
      :radius 0.5
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 0.5
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text "surf"
      :radius 0.5
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :tags "surf"
      :radius 0.5
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str (:name spot) ", " (:name country))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :tags "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :text (str (:name spot) ", " (:name country))
      :tags "surf"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "surf"
      :sort "relevance"}]))

(defn parse-location
  "Parse the Flickr location response."
  [{:keys [latitude longitude]}]
  (when (and latitude longitude)
    (point 4326 (Double/parseDouble longitude) (Double/parseDouble latitude))))

(defn parse-size
  "Parse the Flickr photo size response. The Flickr API returns width
  and height sometimes a string, sometimes as an integer."
  [{:keys [width height] :as size}]
  (cond-> size
    (string? width)
    (update :width #(Long/parseLong %))
    (string? height)
    (update :height #(Long/parseLong %))))

(defn label-name
  "Return the lower cased and trimmed label for `s`."
  [s]
  (some-> s name str/lower-case str/trim))

(defn label-size
  "Return the size of `photo` used for labeling."
  [sizes min-width max-width]
  (let [sorted (sort-by :width sizes)]
    (or (last (filter #(and (>= (:width %) min-width)
                            (<= (:width %) max-width))
                      sorted))
        (first (filter #(> (:width %) max-width) sorted))
        (last (filter #(< (:width %) min-width) sorted)))))

(defn log-labels
  "Log the `labels` from the Vision API."
  [labels]
  (log/infof "  Vision API Labels:  %s"
             (str/join ", "(map :description labels))))

(defn log-search-criteria
  "Log the Flickr search criteria."
  [{:keys [lat lon text radius tags sort]} photos]
  (log/infof "Found %s Flickr photos for search criteria:" (count photos))
  (when (and lat lon) (log/infof "  Location: %s, %s" lat lon))
  (when text (log/infof "  Text:     %s" text))
  (when radius (log/infof "  Radius:   %s km" radius))
  (when tags (log/infof "  Tags:     %s" tags))
  (when sort (log/infof "  Sort:     %s" sort))
  (doseq [photo photos] (log/infof "  URL:      %s" (flickr/photo-url photo))))

(defn save-photo-label-scores!
  "Save the scores `labels` of the Flickr `photo`."
  [db photo labels]
  (when-not (empty? labels)
    (let [values (map #(vector (:id photo) (:mid %) (:score %)) labels)]
      @(sql/insert db :photo-label-scores [:photo-label-id :photo-id :score]
         (sql/select db [:photo-labels.id :v.photo-id :v.score]
           (sql/from (sql/as (sql/values db values)
                             :v [:photo-id :mid :score]))
           (sql/join :photo-labels '(on (= :photo-labels.mid :v.mid))))
         (sql/on-conflict [:photo-label-id :photo-id]
           (sql/do-update {:score :EXCLUDED.score}))))))

(defn save-photo-labels!
  "Save the labels of the Flickr `photo` to `db`."
  [db labels]
  @(sql/insert db :photo-labels []
     (sql/values (map #(select-keys % [:description :mid]) labels))
     (sql/on-conflict [:mid]
       (sql/do-update {:description :EXCLUDED.description}))))

(defn- save-flickr-photo!
  "Save the Flickr `photo` to `db`."
  [db photo]
  (when photo
    (flickr/save-photo! db photo)
    (flickr/save-photo-sizes! db photo (:sizes photo))
    (flickr/save-user-info! db (:user-info photo))))

(defn save-photo!
  "Save the Flickr `photo` of `spot`."
  [db spot photo]
  (let [saved (photos/save! db photo)]
    (spots/add-photo! db spot saved)
    (assert (:id saved) "Photo id expected!")
    (->> (assoc (:flickr photo) :photo-id (:id saved))
         (save-flickr-photo! db))
    (let [labels (:labels photo)]
      (save-photo-labels! db labels)
      (save-photo-label-scores! db saved labels)
      (log/infof "Saved photo #%d \"%s\" for spot #%d \"%s\"."
                 (:id saved) (:title saved)
                 (:id spot) (:name spot))
      (log/infof "  Flickr source URL:  %s" (:url saved))
      (log-labels labels))
    (merge photo saved)))

(defn exclude-existing-photos
  "Return a transducer that removes already saved photos."
  [db]
  (remove #(first (photos/by-url db (:url %)))))

(defn size-by-name
  "Find the `size` of `name`."
  [sizes name]
  (let [name (str/lower-case name)]
    (first (filter #(= (str/lower-case (:label %)) name) sizes))))

(defn search-photos-by-criteria
  "Search photos of `spot` on Flickr by `criteria`."
  [db flickr spot criteria]
  (let [photos (flickr/search-photos flickr criteria)]
    (log-search-criteria criteria photos)
    (->> (into [] (exclude-existing-photos db) photos)
         (pmap (fn [photo]
                 (let [sizes (flickr/photo-sizes flickr (:id photo))]
                   {:flickr (assoc photo :sizes sizes)
                    :label-size (label-size (mapv parse-size sizes) 300 800)
                    :spot (select-keys spot [:id :name])
                    :title (:title photo)
                    :url (:source (size-by-name sizes "original"))}))))))

(defn download-image
  "Download the image used for labeling."
  [photo]
  (if-let [url (-> photo :label-size :source)]
    (try (->> {:as :stream ;; :byte-array doesn not work :/
               :throw-exceptions true}
              (http/get url)
              :body
              slurp-byte-array
              (assoc photo :bytes))
         (catch Exception e
           (log/errorf "Can't download image. HTTP status %s."
                       (-> e ex-data :status))))
    (log/error "Can't download image. No appropriate size found.")))

(defn label-photos
  "Label the `photos` using the Google Vision API."
  [vision-service spot photos]
  (mapv (fn [photo {:keys [labels] :as response}]
          (log/infof "Labeled photo \"%s\" for spot #%d \"%s\"."
                     (:title photo) (:id spot) (:name spot))
          (log/infof "  Flickr source URL:  %s" (:url photo))
          (log-labels labels)
          (assoc photo :labels (vec labels)))
        photos
        (vision/annotate-images
         vision-service (map :bytes photos)
         {:labels 10})))

(defn label-photos-in-batches
  "Returns a transducer that downloads and labels photos."
  [vision spot]
  (comp (partition-all 10)
        (mapcat #(label-photos vision spot %))
        (map #(dissoc % :bytes))))

(defn update-photo-details
  "Fetch the details of the Flickr `photo` and return an updated photo."
  [flickr photo]
  (let [details (flickr/photo-info flickr (-> photo :flickr :id))]
    (-> (assoc photo :location (parse-location (:location details)))
        (assoc-in [:flickr :details] details))))

(defn update-photo-owner
  "Fetch the owner of the Flickr `photo` and return an updated photo."
  [flickr photo]
  (let [user-info (flickr/user-info flickr (-> photo :flickr :owner))]
    (assoc-in photo [:flickr :user-info] user-info)))

(defn photo-label-set
  "Return the labels of `photo` as a set."
  [photo]
  (set (map (comp label-name :description)
            (-> photo :labels))))

(defn include-photos-labeled
  "Include photos that are labeled with any of `labels`."
  [labels]
  (let [labels (set (map label-name labels))]
    (remove #(empty? (set/intersection (photo-label-set %1) labels)))))

(defn exclude-photos-labeled
  "Exclude photos that are labeled with any of `labels`."
  [labels]
  (let [labels (set (map label-name labels))]
    (filter #(empty? (set/intersection (photo-label-set %1) labels)))))

(defn make-pipeline
  [db flickr vision spot max-photos]
  (comp (mapcat #(search-photos-by-criteria db flickr spot %1))
        (distinct-by #(-> % :flickr :id))
        (exclude-existing-photos db)
        (map download-image)
        (remove nil?)
        (label-photos-in-batches vision spot)
        (include-photos-labeled include-labels)
        (exclude-photos-labeled exclude-labels)
        (take max-photos)
        (map (partial update-photo-details flickr))
        (map (partial update-photo-owner flickr))
        (map (partial save-photo! db spot))))

(defn import-photos
  "Import Flickr photos for `spot`."
  [{:keys [db flickr vision] :as env} spot & [{:keys [max-photos]}]]
  (when-let [spot (spots/by-id db (:id spot))]
    (let [country (countries/by-spot db spot)
          search-criterias (search-criterias country spot)
          pipeline (make-pipeline db flickr vision spot (or max-photos 3))
          photos (into [] pipeline search-criterias)]
      (doseq [photo photos]
        (photo-service/download! (:photos env) photo))
      (some->> (first photos) (spots/save-cover-photo! db spot))
      photos)))

(defn resize-photo!
  "Resize the spot `photo`."
  [env photo]
  (photo-service/resize! (:photos env) photo photo-service/dimensions))

(defn- events [env builder]
  (.stream builder (-> env :config :input :events)))

(defn- spot-created
  "Returns a stream of spot-created event."
  [events]
  (events/by-name events :burningswell.api.events/spot-created))

(defn make-topology
  "Make the spot photo worker topology."
  [env]
  (k/with-build-stream builder
    (let [events (events env builder)]
      (-> (spot-created events)
          (k/flat-map-vals #(import-photos env {:id (:spot-id %)}))
          (k/map-vals #(resize-photo! env %))
          (.to (-> env :config :output :created))))))

(defcomponent Worker [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (k/start-topology (k/props config) topology) #(.close %))

(defn worker [& [opts]]
  (map->Worker {:config (config opts)}))
