(ns burningswell.streams.weather.datasources
  (:require [burningswell.db.weather.datasources :as datasources]
            [burningswell.db.weather.variables :as variables]
            [burningswell.streams.core :as k]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [netcdf.dataset :as dataset]
            [peripheral.core :as p :refer [defcomponent]]
            [taoensso.timbre :as log]
            [burningswell.db.weather.models :as models]))

(defn config [& [opts]]
  (->> {:application.id "update-weather-datasources"
        :input {:commands "burningswell.weather.model.update-succeeded"}
        :output {:succeeded "burningswell.weather.datasource.update-succeeded"}}
       (merge opts)))

(defn- datasets
  "Fetch the datasets of the `datasource` and `variables`."
  [datasource variables]
  (try (dataset/with-grid-dataset [grid (:dods datasource)]
         (->> (for [valid-time (dataset/valid-times grid)
                    variable variables]
                {:datasource-id (:id datasource)
                 :valid-time valid-time
                 :variable-id (:id variable)})
              doall not-empty))
       (catch Throwable e
         (throw (ex-info (str "Can't read datasource: " (.getMessage e))
                         {:type ::read-datasource-error
                          :datasource (select-keys datasource [:id :dods])}
                         e)))))

(defn- insert-datasets! [db datasource datasets]
  (->> @(sql/insert db :weather.datasets []
          (sql/values datasets)
          (sql/on-conflict [:datasource-id :variable-id :valid-time]
            (sql/do-nothing))
          (sql/returning :*))
       (assoc datasource :datasets)))

(defn update-datasource!
  "Update the datasets of the `datasource`."
  ([db datasource]
   (->> (variables/by-datasource db datasource)
        (update-datasource! db datasource)))
  ([db datasource variables]
   (if-let [datasets (datasets datasource variables)]
     (let [{:keys [datasets model-id] :as datasource}
           (insert-datasets! db datasource datasets)
           valid-times (sort (map :valid-time datasets))
           model (models/by-id db model-id)]
       (log/info {:msg "Updated weather datasource."
                  :model (select-keys model [:id :name])
                  :reference-time (:reference-time datasource)
                  :variables (mapv :name variables)
                  :start (first valid-times)
                  :end (last valid-times)})
       datasource)
     (assoc datasource :datasets []))))

(s/def ::update-datasource-model
  (s/cat :db sql/db?
         :datasource ::datasources/datasource))

(s/def ::update-datasource-model-variables
  (s/cat :db sql/db?
         :datasource ::datasources/datasource
         :variables (s/coll-of ::variables/variable)))

(s/fdef update-datasource!
  :args (s/or :model ::update-datasource-model
              :model+variables ::update-datasource-model-variables))

;; Kafka Streams

(defn make-topology [{:keys [db config] :as env}]
  (k/with-build-stream builder
    (-> (.stream builder (-> config :input :commands))
        (k/flat-map-vals #(:datasources %))
        (k/map-vals #(update-datasource! db %1))
        (.to (-> config :output :succeeded)))))

(defcomponent Worker [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (k/start-topology (k/props config) topology) #(.close %))

(defn worker [& [opts]]
  (map->Worker {:config (config opts)}))
