(ns burningswell.streams.weather.models
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather.models :as models]
            [burningswell.kafka.producer :as producer]
            [burningswell.streams.core :as k]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [commandline.core :as cli]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [netcdf.dods :as dods]
            [no.en.core :refer [split-by-comma]]
            [peripheral.core :as p :refer [defcomponent]]
            [taoensso.timbre :as log]))

(defn config [& [opts]]
  (->> {:application.id "update-weather-model"
        :input {:commands "burningswell.weather.model.update"}
        :output {:succeeded "burningswell.weather.model.update-succeeded"}}
       (merge opts)))

(defn- latest-reference-time
  [datasources]
  (last (sort (map :reference-time datasources))))

(defn update-latest-reference-time!
  "Update the latest reference time of `model`."
  [db {:keys [datasources] :as model}]
  (let [reference-time (latest-reference-time datasources)]
    (->> (first @(sql/update db :weather.models
                   {:latest-reference-time reference-time}
                   (sql/where `(= :id ~(:id model)))
                   (sql/returning :*)))
         (merge model))))

(defn- datasource->row
  "Return a data source row for `model` and `datasource`."
  [model datasource]
  (-> (select-keys datasource [:das :dds :dods :reference-time])
      (assoc :model-id (:id model))))

(defn insert-datasources!
  "Update the data sources of `model`."
  [db  {:keys [datasources] :as model}]
  (->> @(sql/insert db :weather.datasources []
          (sql/values (map #(datasource->row model %) datasources))
          (sql/on-conflict [:model-id :reference-time]
            (sql/do-nothing))
          (sql/returning :*))
       (assoc model :datasources)))

(s/fdef insert-datasources!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

(defn update-model!
  "Update the data sources and the latest reference time of `model`."
  [db model]
  (let [{:keys [datasources] :as model}
        (->> (assoc model :datasources (dods/datasources model))
             (update-latest-reference-time! db)
             (insert-datasources! db))]
    (log/info {:msg "Updated weather model."
               :model (select-keys model [:id :name])
               :datasources (count datasources)})
    model))

(s/fdef update-model!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

;; Kafka Streams

(defn make-topology [{:keys [db config] :as env}]
  (k/with-build-stream builder
    (-> (.stream builder (-> config :input :commands))
        (k/map-vals #(update-model! db %))
        (.to (-> config :output :succeeded)))))

(defcomponent Worker [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (k/start-topology (k/props config) topology) #(.close %))

(defn worker [& [opts]]
  (map->Worker {:config (config opts)}))

;; Command line client

(defn client
  "Returns a client for the weather dataset worker."
  [{:keys [db producer] :as config}]
  (component/system-map
   :db (db/new-db db)
   :producer (producer/producer producer)))

(defn cli-config [env]
  {:db (config/db env)
   :producer (config/kafka env)})

(defn- cli-models
  "Returns the models for the client."
  [db models]
  (if-let [models (not-empty (split-by-comma models))]
    (models/by-names db models)
    (models/all db)))

(defn run
  "Run the client of the weather model worker."
  [{:keys [producer db] :as system} & args]
  (cli/with-commandline [[opts variables] args]
    [[h help "Print this help."]
     [m models "The weather models to load." :string "MODELS"]]
    (if (:help opts)
      (cli/print-help "weather-models [OPTION...]")
      (doseq [model (cli-models db (:models opts))]
        (producer/send! producer (-> (config) :input :commands) nil model)
        (log/info {:msg "Queued weather model to update inventory."
                   :model model})))))

(defn -main
  "Publish commands to update the data sources of the weather models."
  [& args]
  (p/with-start [system (client (cli-config env))]
    (apply run system args)))

(comment
  (-main "-m" "gfs")
  (-main "-m" "nww3")
  (-main "-m" "akw")
  (-main))
