(ns burningswell.streams.weather.spots
  (:require [burningswell.db.weather.datasets :as datasets]
            [burningswell.streams.core :as k]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :as t]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [peripheral.core :as p :refer [defcomponent]]
            [taoensso.timbre :as log]))

(defn config [& [opts]]
  (->> {:application.id "update-weather-spots"
        :input {:commands "burningswell.weather.dataset.update-succeeded"}
        :output {:succeeded "burningswell.weather.spots.update-succeeded"}}
       (merge opts)))

(defn partition-table
  "Returns the partition table name."
  [valid-time]
  (let [valid-time (to-date-time valid-time)]
    (keyword (format "weather.spots-%04d-%02d"
                     (t/year valid-time)
                     (t/month valid-time)))))

(s/fdef partition-table
  :args (s/cat :valid-time inst?)
  :ret keyword?)

(defn partition-start-time
  "Returns the partition start time for `valid-time`."
  [valid-time]
  (-> valid-time (.withDayOfMonth 1) (.withTime 0 0 0 0)))

(defn partition-end-time
  "Returns the partition end time for `valid-time`."
  [valid-time]
  (t/plus (partition-start-time valid-time) (t/months 1)))

(defn create-partition-table!
  "Create the raster partition table for the `dataset` in `db`."
  [db {:keys [valid-time] :as dataset}]
  (let [partition (partition-table valid-time)
        valid-time (to-date-time valid-time)
        start-time (partition-start-time valid-time)
        end-time (partition-end-time valid-time)]
    @(sql/create-table db partition
       (sql/if-not-exists true)
       (sql/check `(and (>= :valid-time (cast ~start-time :timestamptz))
                        (< :valid-time (cast ~end-time :timestamptz))))
       (sql/like :weather.spots :including [:all])
       (sql/inherits :weather.spots))
    (assoc dataset :partition-table partition)))

(s/fdef create-partition-table!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn drop-partition-table!
  "Drop the spot weather partition table for the `dataset` in `db`."
  [db {:keys [valid-time] :as dataset}]
  @(sql/drop-table db [(partition-table valid-time)]
     (sql/if-exists true)))

(s/fdef drop-partition-table!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset))

(defn select-spot-weather
  "Select the spot weather for `dataset`."
  [db dataset]
  (let [spot-location '(st_transform (cast :spots.location :geometry) 3395)]
    (sql/select db (sql/distinct
                    [(sql/as :spots.id :spot-id)
                     :model-id
                     :variable-id
                     :reference-time
                     :valid-time
                     (sql/as `(cast
                               (coalesce
                                (st_value :rast 1 ~spot-location)
                                (avg_neighborhood
                                 (st_neighborhood
                                  :rast 1 ~spot-location 1 1)))
                               :double-precision)
                             :value)]
                    :on [:spots.id :variable-id :valid-time])
      (sql/from :weather.rasters)
      (sql/join :spots `(on (st_intersects :rast ~spot-location)))
      (sql/where `(and (= :rasters.valid-time ~(:valid-time dataset))
                       (= :rasters.variable-id ~(:variable-id dataset))))
      (sql/order-by :spots.id :variable-id :valid-time
                    (sql/desc '(abs (* (st_scalex :rast)
                                       (st_scaley :rast))))))))

(s/fdef select-spot-weather
  :args (s/cat :db sql/db? :dataset ::datasets/dataset))

(defn update-partition!
  "Update the spot weather partition for `dataset` in `db`."
  [db dataset]
  (->> @(sql/insert db (partition-table (:valid-time dataset))
            [:spot-id :model-id :variable-id :reference-time :valid-time :value]
          (select-spot-weather db dataset)
          (sql/on-conflict [:spot-id :variable-id :valid-time]
            (sql/do-update
             {:model-id :EXCLUDED.model-id
              :reference-time :EXCLUDED.reference-time
              :value :EXCLUDED.value})))
       first :count
       (assoc dataset :count)))

(s/fdef update-partition!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret nat-int?)

(defn update-spot-weather!
  "Download the `dataset`, convert and center it, and finally load it
  into the `db`."
  [db {:keys [valid-time] :as dataset}]
  (create-partition-table! db dataset)
  (let [dataset (update-partition! db dataset)]
    (log/info {:msg "Updated spot weather."
               :dataset (datasets/by-id db (:id dataset))})
    dataset))

(s/fdef update-spot-weather!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

;; Kafka Streams

(defn make-topology [{:keys [db config] :as env}]
  (k/with-build-stream builder
    (-> (.stream builder (-> config :input :commands))
        (k/map-vals #(update-spot-weather! db (:dataset %1)))
        (.to (-> config :output :succeeded)))))

(defcomponent Worker [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (k/start-topology (k/props config) topology) #(.close %))

(defn worker [& [opts]]
  (map->Worker {:config (config opts)}))
