(ns burningswell.transit
  (:require #?(:clj [clj-time.coerce :refer [to-long]])
            #?(:clj [schema.utils])
            [cognitect.transit :as transit]
            [geo.transit :as geo-transit])
  #?(:clj (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
                   [org.joda.time DateTime]
                   [schema.utils ValidationError])))

(def read-validation-error
  "Read a validation error."
  (transit/read-handler
   (fn [[srid coordinates]]
     nil)))

#?(:clj (def write-class
          "Write a Class as a symbol"
          (transit/write-handler
           (constantly "$")
           (fn [class] (str class))
           (constantly nil))))

#?(:clj (def write-date-time
          "Write org.joda.time.DateTime."
          (transit/write-handler
           (constantly "m")
           (fn [date-time]
             (str (to-long date-time)))
           (constantly nil))))

#?(:clj (def write-validation-error
          "Write a validation error."
          (transit/write-handler
           (constantly "burningswell/validation-error")
           (fn [error]
             (schema.utils/validation-error-explain error))
           (constantly nil))))

(def read-handlers
  "The Transit read handlers."
  (merge
   geo-transit/read-handlers
   #?(:clj {"burningswell/validation-error" read-validation-error})))

(def write-handlers
  "The Transit write handlers."
  (merge
   geo-transit/write-handlers
   #?(:clj {Class write-class
            DateTime write-date-time
            ValidationError write-validation-error})))

(defn encode
  "Encode `payload` into Transit format."
  [payload & [opts]]
  (let [opts (merge {:handlers write-handlers} opts)]
    #?(:clj (let [format (or (:format opts) :json)
                  stream (ByteArrayOutputStream.)
                  writer (transit/writer stream format opts)
                  _ (transit/write writer payload)]
              (case format
                :json (.toString stream)
                :msgpack (.toByteArray stream)))
       :cljs (transit/write (transit/writer :json opts) payload))))

(defn decode
  "Decode `payload` from Transit format."
  [payload & [opts]]
  (let [opts (merge {:handlers read-handlers} opts)]
    #?(:clj (let [format (or (:format opts) :json)
                  stream (ByteArrayInputStream.
                          (case format
                            :json (.getBytes payload)
                            :msgpack payload))
                  reader (transit/reader stream format opts)]
              (transit/read reader))
       :cljs (transit/read (transit/reader :json opts) payload))))
