(ns burningswell.weather.web
  (:require [burningswell.weather.db :as db]
            [burningswell.transit :as transit]
            [burningswell.json :as json]
            [hiccup.element :refer [javascript-tag]]
            [hiccup.page :refer [html5 include-css include-js]]
            [clojure.pprint :as pprint]
            [yada.yada :as yada]
            [schema.core :as s]
            [geo.postgis :as geo]))

(def default-location
  "The default location, Uluwatu, Bali, Indonesia."
  (geo/point 4326 115.08598373601123 -8.816363902091963))

(def highlight-css
  "The highlight.js stylesheet."
  (str "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.2.0"
       "/styles/solarized-light.min.css"))

(def highlight-js
  "The highlight.js JavaScript code."
  "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.2.0/highlight.min.js")

(defmulti render
  "Render `data` in HTML format."
  (fn [context data]
    (keyword (yada/content-type context))))

(defmethod render :application/edn [context data]
  (pr-str data))

(defmethod render :application/json [context data]
  (json/json-str data))

(defmethod render :application/transit+json [context data]
  (transit/encode data :json))

(defmethod render :application/transit+msgpack [context data]
  (transit/encode data :msgpack))

(defn fade-in-key-frames [from to]
  (format
   "@keyframes fadein {
     from { opacity: %d; }
     to { opacity: %d; }
    }" from to))

(defmethod render :text/html [context data]
  (html5
   {:style "background-color: #fdf6e3;"}
   [:head
    (include-js highlight-js)
    (javascript-tag "hljs.initHighlightingOnLoad();")
    (include-css highlight-css)
    [:style (fade-in-key-frames 0 1)]]
   [:body
    {:style "animation: fadein 2s;"}
    [:pre [:code (with-out-str (pprint/pprint data))]]]))

(defmethod render :text/plain [context data]
  (with-out-str (pprint/pprint data)))

(defn get-query
  "Get the query parameters of the `context`."
  [context]
  (get-in context [:parameters :query]))

(defn get-location
  "Get the location from the `context` query parameters."
  [context]
  (let [{:keys [latitude longitude]} (get-query context)]
    (when (and latitude longitude)
      (geo/point 4326 longitude latitude))))

(defn forecasts [db]
  (yada/resource
   {:description "Phonebook entry"
    :parameters
    {:query
     {(s/optional-key :latitude) Long
      (s/optional-key :longitude) Long}}
    :produces
    [{:media-type
      #{"application/edn;q=0.9"
        "application/json;q=0.8"
        "application/transit+json"
        "application/transit+msgpack"
        "text/html"
        "text/plain"}
      :charset "UTF-8"}]
    :methods
    {:get
     {:response
      (fn [context]
        (let [location (or (get-location context) default-location)]
          (println location)
          (render context (db/weather-forecast db location))))}}}))
