(ns burningswell.weather.import
  (:gen-class)
  (:refer-clojure :exclude [distinct group-by replace update])
  (:require [again.core :as again]
            [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather :as weather]
            [burningswell.db.weather.datasets :as ds]
            [burningswell.db.weather.models :as models]
            [burningswell.db.weather.variables :as variables]
            [burningswell.file :refer [with-tmp-files]]
            [burningswell.system :refer [defsystem]]
            [burningswell.weather.gdal :as gdal]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :refer [day hour month now year]]
            [clojure.java.io :as io]
            [clojure.java.jdbc :as jdbc]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer :all]
            [datumbazo.shell :as shell :refer [raster2pgsql]]
            [datumbazo.util :refer [exec-sql-file]]
            [environ.core :refer [env]]
            [netcdf.dataset :as dataset]
            [netcdf.dods :as dods]
            [no.en.core :refer [split-by-comma]]))

(defsystem system
  "The netcdf CLI system."
  [{:keys [db]}]
  (component/system-map :db (db/new-db db)))

(defn variables-and-models
  "Return the variables and their models from `config`."
  [db {:keys [models variables] :as config}]
  (let [models (set (models/by-names-or-all db models))]
    (for [variable (variables/by-names-or-all db variables)]
      [variable (filter models (models/by-variable db variable))])))

(defn netcdf-file
  "Return the dataset filename for the `variable` in `model` at
  `reference-time`."
  [directory variable model reference-time]
  (io/file
   directory
   (:name model)
   (:name variable)
   (format "%04d" (year reference-time))
   (format "%02d" (month reference-time))
   (format "%02d" (day reference-time))
   (format "%02d.nc" (hour reference-time))))

(defn netcdf-md5-file
  "Return the dataset filename for the `variable` in `model` at
  `reference-time`."
  [directory variable model reference-time]
  (-> (netcdf-file directory variable model reference-time)
      (str ".md5")
      (io/file)))

(defn netcdf-sql-file
  "Return the raster SQL filename for the `variable` in `model` at
  `reference-time`."
  [directory variable model reference-time]
  (-> (netcdf-file directory variable model reference-time)
      (str/replace #".nc" ".sql")
      (io/file)))

(defn virtual-format-file
  "Return the VRT filename for the `variable` in `model` at
  `reference-time`."
  [directory variable model reference-time]
  (-> (netcdf-file directory variable model reference-time)
      (str/replace #".nc" ".vrt")
      (io/file)))

(defn raster-table
  "Returns the temporary raster table for a dataset."
  [model variable reference-time]
  (keyword (format "%s-%s-%04d-%02d-%02d-%02d"
                   (:name model) (:name variable)
                   (year reference-time)
                   (month reference-time)
                   (day reference-time)
                   (hour reference-time))))

(defn datasets [system variable model reference-time]
  (again/with-retries [100 1000 10000]
    (let [reference-time (or reference-time (now))]
      (if-let [datasource (dods/datasource model reference-time)]
        (dataset/with-grid-dataset [grid (:dods datasource)]
          (doall (for [valid-time (dataset/valid-times grid)]
                   {:_embedded
                    {:model model
                     :variable variable}
                    :das (:das datasource)
                    :dds (:dds datasource)
                    :dods (:dods datasource)
                    :valid-time valid-time
                    :reference-time (:reference-time datasource)})))
        (throw (ex-info "Can't open dataset."
                        {:model model
                         :variable variable
                         :reference-time reference-time}))))))

(defn delete-dataset [db dataset]
  (delete db :weather.rasters
    (where `(= :dataset-id ~(:id dataset)))))

(defn insert-dataset [db dataset table band]
  (let [column (keyword (str (name table) ".rast"))]
    (insert db :weather.rasters [:dataset-id :rast]
      (select db [(as (:id dataset) :dataset-id)
                  `(st_band ~column ~band)]
        (from table)
        (join :weather.rasters
              `(on (= :weather.rasters.dataset-id
                      ~(:id dataset)))
              :type :left)
        (where '(is-null :weather.rasters.id))))))

(defn create-virtual-format [input output]
  (with-tmp-files [tmp]
    (gdal/create-virtual-format input tmp)
    (gdal/center-virtual-format-file tmp output)))

(defn import-dataset
  [system variable model reference-time]
  (let [datasets (datasets system variable model reference-time)]
    (log/infof "Downloading dataset:")
    (log/infof "  Variable .......... %s (%s)"
               (:description variable) (:name variable))
    (log/infof "  Model ............. %s (%s)"
               (:description model) (:name model))
    (log/infof "  Reference time .... %s"
               (-> datasets first :reference-time))
    (log/infof "  Time range  ....... %s - %s"
               (-> datasets first :valid-time)
               (-> datasets last :valid-time))
    (when (not-empty datasets)
      (let [db (:db system)
            reference-time (-> datasets first :reference-time)
            directory "netcdf"
            netcdf-file (netcdf-file directory variable model reference-time)
            md5-file (netcdf-md5-file directory variable model reference-time)
            sql-file (netcdf-sql-file directory variable model reference-time)
            virtual-format (virtual-format-file
                            directory variable
                            model reference-time)
            tmp-table (raster-table model variable reference-time)]
        (when (not (.exists md5-file))
          (io/make-parents netcdf-file)
          (dataset/copy-dataset (-> datasets first :dods)
                                netcdf-file [(:name variable)]))
        (create-virtual-format netcdf-file virtual-format)
        (raster2pgsql db tmp-table virtual-format sql-file
                      {:constraints true
                       :height 10
                       :mode :create
                       :padding true
                       :regular-blocking true
                       :srid 4326
                       :width 10})
        @(drop-table db [tmp-table] (if-exists true))
        (exec-sql-file db sql-file)
        (jdbc/with-db-transaction [db db]
          (doseq [[index dataset] (map-indexed vector datasets)
                  :let [band (inc index)
                        dataset (ds/save db dataset)]]
            @(delete-dataset db dataset)
            @(insert-dataset db dataset tmp-table band)))
        @(drop-table db [tmp-table])))
    datasets))

(defn import-variable
  [system variable models reference-time]
  (let [import #(import-dataset system variable % reference-time)]
    (doall (mapcat import models))))

(defn update-forecast
  [{:keys [db] :as system} datasets]
  (let [datasets  (apply concat datasets)
        valid-times (sort (map :valid-time datasets))
        start (first valid-times)
        end (last valid-times)]
    (log/infof "Loading weather data: %s - %s." start end)
    @(weather/import-spot-forecasts db {:start start :end end})
    (log/info "Successfully loaded weather data.")
    (log/info "Refreshing 3-hourly materialized weather view.")
    @(weather/refresh-3-hourly-spot-weather db)
    (log/info "Successfully refreshed materialized view.")
    datasets))

(defn run-cli
  [{:keys [models variables reference-time] :as config}]
  (with-system [{:keys [db] :as system} config]
    (doseq [[variable models] (variables-and-models db config)
            :when (and variable (not-empty models))]
      (import-variable system variable models reference-time))))

(defn -main [& args]
  (with-commandline [variables args]
    [[h help "Print this help."]
     [m models "The weather models to load." :string "MODELS"]
     [r reference-time "The reference time to load." :time "REFERENCE-TIME"]]
    (when help
      (print-help "weather-import [OPTION...] [VARIABLE,...]")
      (System/exit 0))
    (run-cli {:db (config/db env)
              :models (set (split-by-comma models))
              :reference-time (to-date-time reference-time)
              :variables (set variables)})))

(comment
  (weather/import-spot-forecasts (db))
  (-main)
  (-main "-m" "nww3" "htsgwsfc")
  (-main "-m" "gfs"))
