(ns burningswell.weather.db
  (:refer-clojure :exclude [distinct group-by update])
  (:require [datumbazo.core :refer :all]
            [clj-time.coerce :refer [to-date-time]]
            [medley.core :as medley]))

(defn- select-vrd-interval
  "Select the interval between :valid-time of
  each :variable-id, :reference-time combination."
  [db]
  (select db [:variable-id
              :reference-time
              (as '(- ((lag :valid-time) over
                       (partition by :variable-id
                                  order by :valid-time desc))
                      :valid-time)
                  :duration)]
    (from :weather.datasets)
    (where `(>= :datasets.valid-time (now)))))

(defn- select-avg-vrd-interval
  "Select the average interval between :valid-time of
  each :variable-id, :reference-time combination."
  [db]
  (select db [:intervals.variable-id
              :intervals.reference-time
              (as '(max :intervals.duration) :duration)]
    (from (as (select-vrd-interval db) :intervals))
    (group-by :intervals.variable-id :intervals.reference-time)))

(defn- select-avg-valid-time-interval
  "Select the average interval between :valid-time."
  [db]
  (select db [(as '(min :interval.duration) :duration)]
    (from (as (select-avg-vrd-interval db) :interval))))

(defn- select-current-forecast-time
  "Return the current forecast time."
  [db interval]
  (select db [(as '(min :datasets.valid-time) :valid-time)]
    (from :weather.datasets)
    (where `(>= :datasets.valid-time
                (- (now)
                   (cast ~(or interval "3 hours") :interval))))
    (group-by :datasets.variable-id)))

(defn current-forecast-time
  "Return the current forecast time."
  [db]
  (when-let [interval (-> @(select-avg-valid-time-interval db) first :duration)]
    (-> @(select-current-forecast-time db interval)
        first :valid-time)))

(defn select-weather-forecast [db location time]
  (select db (distinct
              [(as '(json_build_object
                     "id" :models.id
                     "name" :models.name
                     "description" :models.description
                     "reference-time" :datasets.reference-time)
                   :model)
               (as `(json_build_object
                     "id" :variables.id
                     "name" :variables.name
                     "description" :variables.description
                     "valid-time" :datasets.valid-time
                     "value" (round (cast (avg_neighborhood
                                           (st_neighborhood
                                            :rast 1 ~location 1 1))
                                          :numeric)
                                    4)
                     "unit" :variables.unit)
                   :variable)]
              :on [:datasets.variable-id :datasets.valid-time])
    (from :weather.datasets)
    (join :weather.rasters.dataset-id :weather.datasets.id)
    (join :weather.variables.id :weather.datasets.variable-id)
    (join :weather.models.id :weather.datasets.model-id)
    (where `(and (st_intersects :rast ~location)
                 (= :datasets.valid-time ~time)))
    (order-by :datasets.variable-id :datasets.valid-time
              (desc '(abs (* (st_scalex :rast)
                             (st_scaley :rast)))))))

(defn weather-forecast [db location]
  (when-let [time (current-forecast-time db)]
    (->> @(select-weather-forecast db location time)
         (map #(update-in % [:model :reference-time] to-date-time))
         (map #(update-in % [:variable :valid-time] to-date-time))
         (clojure.core/group-by #(get-in % [:variable :valid-time])))))

(comment

  (clojure.pprint/print-table
   @(select-avg-valid-time-interval (-> reloaded.repl/system :db)))

  (current-forecast-time (-> reloaded.repl/system :db))
  (weather-forecast
   (-> reloaded.repl/system :db)
   (geo.postgis/point 4326 115.08598373601123 -8.816363902091963)))
