(ns burningswell.weather.forecast
  "Create the Burning Swell weather forecast from the raster data."
  (:gen-class)
  (:refer-clojure :exclude [distinct group-by replace update])
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather :as weather]
            [burningswell.system :refer [defsystem]]
            [burningswell.time :refer [current-duration-str]]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :as time]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer :all]
            [environ.core :refer [env]]
            [no.en.core :refer [parse-long]]))

(defsystem system
  "The weather forecast CLI system."
  [{:keys [db]}]
  (component/system-map :db (db/new-db db)))

(defn- parse-spots
  "Parse the command line spot ids."
  [spots]
  (->> (map parse-long spots)
       (remove nil?)
       (map #(hash-map :id %))))

(defn select-time-range
  "Select the weather forecast start and end time."
  [db]
  (some-> @(select db [(as '(min :datasets.reference-time) :start)
                       (as '(max :datasets.valid-time) :end)]
             (from :weather.datasets)
             (join :weather.spot-forecasts.reference-time
                   :weather.datasets.reference-time
                   :type :left)
             (where '(is-null :weather.spot-forecasts.reference-time)))
          (first)
          (update-in [:start] to-date-time)
          (update-in [:end] to-date-time)))

(defn- spot-count
  "Return the spot count, either from the `opts` or the `db`."
  [db {:keys [spots]}]
  (if (empty? spots)
    (-> @(select db [(as '(count :*) :count)]
           (from :spots))
        first :count)
    (count spots)))

(defn import-weather-forecast
  "Create the weather forecasts for spots."
  [db opts]
  (when-not (:skip-import opts)
    (let [started-at (time/now)]
      (log/info "Building weather forecast ...")
      (when-let [start (:start opts)]
        (log/infof "  Start time ....... %s" start))
      (when-let [end (:end opts)]
        (log/infof "  End time ......... %s" end))
      (log/infof "  Number of spots .. %s" (spot-count db opts))
      (let [result @(weather/import-spot-forecasts db opts)
            row-count (-> result first :count)]
        (log/infof "  Rows imported .... %s" row-count)
        (log/infof "  Duration ......... %s"
                   (current-duration-str started-at))
        (->> {:import-started-at started-at
              :import-finished-at (time/now)
              :import-row-count row-count}
             (merge opts))))))

(defn run-cli
  "Create the Burning Swell weather forecast from the raster data."
  [{:keys [db]} & [{:keys [spots start end] :as opts}]]
  (let [spots (parse-spots spots)
        time-range (select-time-range db)
        start (or start (:start time-range))
        end (or end (:end time-range))]
    (when (or (and start end (time/after? end start))
              (:force opts))
      (->> (assoc opts
                  :end end
                  :spots spots
                  :start start)
           (import-weather-forecast db)))))

(defn -main
  "Create the Burning Swell weather forecast from the raster data."
  [& args]
  (with-commandline [[opts spots] args]
    [[I skip-import "Skip the import of the weather forecast."]
     [R skip-refresh "Skip the refreshing of the weather forecast."]
     [e end "The weather forecast start time." :string "END"]
     [f force "Force import and refresh of the weather forecast."]
     [h help "Print this help."]
     [s start "The weather forecast end time." :time "START"]]
    (if (:help opts)
      (print-help "weather-forecast [OPTION...] [SPOT-ID,...]")
      (with-system [system {:db (config/db env)}]
        (run-cli system (assoc opts :spots spots))))))

(comment
  (def d (db/new-db (config/db env)))

  (-main "-f" "-s" "2000-01-01")
  (-main "-f")
  (-main))
