(ns burningswell.web.infinite-list
  (:require [burningswell.web.mixins.event-handler :as events]
            [cljs.core.async :as async :refer [<! chan close! pub timeout]]
            [goog.events :as gevents]
            [goog.dom :as gdom]
            [goog.style :as style]
            [om-tools.core :refer-macros [defcomponentk]]
            [om.core :as om :include-macros true]
            [sablono.core :as html :refer-macros [html]])
  (:require-macros [cljs.core.async.macros :refer [go]])
  (:import goog.events.EventType))

(defn page-height
  "Return the height of the page."
  []
  js/document.documentElement.clientHeight)

(defn scroller
  "Return the scroller of `event`."
  [owner event]
  (let [target (.-target event)]
    (or (aget target "scroller") target)))

(defn scroll-top
  "Return the top position of the `event` scroller."
  [owner event]
  (.-scrollTop (scroller owner event)))

(defn scroll-content-height
  "Return the height of the `event` scroller."
  [owner event]
  (.-scrollHeight (scroller owner event)))

(defn scroll-header-height
  "Return the header height of the `event` scroller."
  [owner event]
  (if-let [header (.-header (.-target event))]
    (.-height (style/getSize header)) 0))

(defn scroll-visible-height
  "Return the content height of the `event` scroller."
  [owner event]
  (- (page-height) (scroll-header-height owner event)))

(defn content-height
  "Return the height of the content."
  [owner]
  (let [node (om/get-node owner)]
    (.-height (style/getSize node))))

(defn bottom-reached?
  "Returns true if the scroll position is near the bottom of the page."
  [owner event & [threshold]]
  (neg? (- (scroll-content-height owner event)
           (scroll-visible-height owner event)
           (scroll-top owner event)
           (or threshold 20))))

(defn next-page-params
  "Return the params for the next page action."
  [owner]
  (let [{:keys [action-params page per-page query-params]}
        (om/get-state owner)]
    (-> (merge action-params  query-params)
        (assoc :page (inc page) :per-page per-page))))

(defn next-page
  "Fetch the next page."
  [owner]
  (when-not (om/get-state owner :loading)
    (om/set-state! owner :loading true)
    (when-let [action (om/get-state owner :action)]
      (action (om/get-shared owner) (next-page-params owner)))))

(defn on-scroll
  "The scroll event handler."
  [owner event]
  (when (and (not (empty? (om/get-props owner)))
             (bottom-reached? owner event))
    (println "Infinite list bottom reached.")
    (next-page owner)))

(defn on-resize
  "The resize event handler."
  [owner event]
  (when (< (content-height owner) (page-height))
    (next-page owner)))

(defn scroll-target
  "Return the scroll target of `owner`."
  [owner]
  (let [scroll-target (om/get-state owner :scroll-target)]
    (cond
      (nil? scroll-target)
      js/window
      (string? scroll-target)
      (or (some-> (js/document.getElementsByTagName scroll-target)
                  array-seq first)
          (gdom/getElementByClass scroll-target))
      (ifn? scroll-target)
      (scroll-target)
      :else scroll-target)))

(defn attach-listeners
  "Attach scroll and resize listeners."
  [owner scroller]
  (events/listen owner scroller EventType.SCROLL #(on-scroll owner %))
  (events/listen owner scroller EventType.RESIZE #(on-resize owner %))
  (events/listen owner js/window EventType.RESIZE #(on-resize owner %)))

(defcomponentk infinite-list-item
  "An infinite list item component."
  [data owner opts]
  (render [_]
    (html
     [:li.infinite-list__item
      (om/build (:render opts) data)])))

(defcomponentk infinite-list
  "An infinite list component."
  [[:data list scroller] owner state opts]
  (:mixins events/handler)
  (init-state [_]
    (let [per-page (:per-page opts 10)]
      {:action (:action opts)
       :key :id
       :listeners []
       :loading false
       :page (int (/ (count list) per-page))
       :per-page per-page
       :scroll-target (:scroll-target opts)
       :query-params (:query-params opts)}))
  (did-update [_ prev-props prev-state]
    (when (and (not (:loading @state))
               (> (count list) (count prev-props))
               (< (content-height owner) (page-height)))
      (next-page owner)))
  (will-mount [_]
    (if (empty? list)
      (next-page owner)))
  (did-mount [_]
    (when scroller
      (attach-listeners owner scroller)))
  (will-receive-props [_ next-props]
    (when (and (nil? scroller) (:scroller next-props))
      (attach-listeners owner (:scroller next-props)))
    (let [{:keys [loading page]} @state]
      (if loading
        (om/update-state!
         owner #(assoc % :loading false :page (inc page))))))
  (render [_]
    (html
     [:ul.infinite-list
      {:class (:class opts)}
      (om/build-all infinite-list-item list
                    {:key (:key @state)
                     :opts opts})])))
