(ns burningswell.web.service-worker
  (:require [burningswell.web.logging :as log]
            [com.stuartsierra.component :as component]))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.service-worker"))

(def service-worker-url
  "The URL of the service worker JavaScript."
  "/javascripts/service-worker.js")

(defn service-worker?
  "Returns true if service worker is supported, otherwise false."
  []
  (some? (aget js/navigator "serviceWorker")))

(defn register-service-worker
  "Register the service worker at `url`."
  [url]
  (-> (.-serviceWorker js/navigator)
      (.register url)
      (.then
       (fn []
         (log/info logger "Service worker registered."))
       (fn [error]
         (log/error logger "Service worker registration failed.")
         (log/error logger error)))))

(when (service-worker?)
  (register-service-worker service-worker-url))

(defrecord ServiceWorker []
  component/Lifecycle
  (start [worker]
    (when (service-worker?)
      (register-service-worker service-worker-url))
    (log/info logger "Service worker started.")
    worker)
  (stop [worker]
    (log/info logger "Service worker stopped.")
    worker))

(defn service-worker
  "Return a new service worker component."
  [& [opts]]
  (map->ServiceWorker opts))
