(ns burningswell.web.geocoder
  (:require-macros [cljs.core.async.macros :refer [go]])
  (:require [cljs.core.async :refer [<! chan close! put!]]
            [geo.core :as geo]))

(defn point->lat-lng [point]
  (if point
    (google.maps.LatLng.
     (geo/point-y point)
     (geo/point-x point))))

(defn address-component [result component & {:keys [type]}]
  (let [components (get result "address_components")]
    (if-let [component (first (filter #(contains? (set (get %1 "types")) component) components))]
      (get (js->clj component)
           (case type
             :short "short_name"
             "long_name")))))

(defn location [result]
  (if-let [location (get-in result ["geometry" "location"])]
    (geo/point 4326 (.lng location) (.lat location))))

(defn make-address [result]
  (let [result (js->clj result)]
    {:formatted (get-in result ["formatted_address"])
     :location (location result)
     :street-name  (address-component result "route")
     :street-number  (address-component result "street_number")
     :postal-code (address-component result "postal_code")
     :region {:name (address-component result "locality")}
     :country {:name (address-component result "country")
               :iso-3166-1-alpha-2 (address-component result "country" :type :short)}}))

(defn geocode
  "Send the `request` to Google's geocoder and return a channel that
  will contain the response."
  [request]
  (let [channel (chan)]
    (.geocode
     (google.maps.Geocoder.)
     (clj->js request)
     (fn [results status]
       (try (cond
              (= status google.maps.GeocoderStatus.OK)
              (put! channel (map make-address results))
              (= status google.maps.GeocoderStatus.ZERO_RESULTS)
              (put! channel []))
            (finally (close! channel)))))
    channel))

(defn geocode-address
  "Geocode `address` and return a channel with results."
  [address]
  (geocode {:address address}))

(defn geocode-point
  "Geocode `point` and return a channel with results."
  [point]
  (geocode {:latLng (point->lat-lng point)}))
