(ns burningswell.web.stores
  (:require [burningswell.web.errors :as errors]
            [burningswell.web.stores.auto-complete :as auto-complete]
            [burningswell.web.stores.countries :as countries]
            [burningswell.web.stores.map :as map]
            [burningswell.web.stores.search :as search]
            [burningswell.web.stores.country :as country]
            [burningswell.web.stores.region :as region]
            [burningswell.web.stores.regions :as regions]
            [burningswell.web.stores.resources :as resources]
            [burningswell.web.stores.route :as route]
            [burningswell.web.stores.signin :as signin]
            [burningswell.web.stores.signup :as signup]
            [burningswell.web.stores.spot :as spot]
            [burningswell.web.stores.spots :as spots]
            [burningswell.web.stores.page :as page]
            [burningswell.web.stores.weather :as weather]
            [clojure.string :as str]
            [coolant.core :as coolant]
            [geo.core :as geo]
            [hal.core :as hal]
            [hodgepodge.core :as hodgepodge]
            [no.en.core :refer [deep-merge]]))

(defn- merge-resources
  [state resources]
  (reduce
   (fn [state resource]
     (assoc state (:id resource) resource))
   state resources))

(defn- update-input-value
  "Update the text field value and clear the error."
  [state name value]
  (-> (assoc-in state [name :value] value)
      (assoc-in [name :error] nil)))

;; (def user
;;   (coolant/store
;;    :user nil
;;    {:user/local-storage
;;     (fn [state user] user)
;;     :user/received
;;     (fn [state user] user)}))

;; (def signin
;;   (coolant/store
;;    :signin {:login {:value nil}
;;             :password {:value nil}
;;             :remember-me {:value nil}}
;;    {:signin/clear
;;     (fn [state user]
;;       (-> (update-input-value state :credentials nil)
;;           (update-input-value :password nil)))
;;     :signin/errors
;;     (fn [state errors]
;;       (-> (reduce-kv
;;            (fn [state key error]
;;              (->> {:value (get-in state [key :value])
;;                    :error error}
;;                   (errors/message-for key)
;;                   (assoc-in state [key :error])))
;;            state (merge {:login nil :password nil} errors))
;;           (assoc :failed-at (js/Date.))))
;;     :signin/login
;;     (fn [state login]
;;       (update-input-value state :login login))
;;     :signin/password
;;     (fn [state password]
;;       (update-input-value state :password password))
;;     :signin/remember-me
;;     (fn [state remember-me]
;;       (update-input-value state :remember-me remember-me))}))

;; (def signup
;;   (coolant/store
;;    :signup {:username {:value nil}
;;             :email {:value nil}
;;             :password {:value nil}
;;             :terms-of-service {:value nil}}
;;    {:signup/complete
;;     (fn [state user]
;;       (assoc state :errors nil :form {} :user user))
;;     :signup/errors
;;     (fn [state errors]
;;       (-> (reduce-kv
;;            (fn [state key error]
;;              (->> {:value (get-in state [key :value])
;;                    :error error}
;;                   (errors/message-for key)
;;                   (assoc-in state [key :error])))
;;            state errors)
;;           (assoc :failed-at (js/Date.))))
;;     :signup/email
;;     (fn [state email]
;;       (update-input-value state :email email))
;;     :signup/username
;;     (fn [state username]
;;       (update-input-value state :username username))
;;     :signup/password
;;     (fn [state password]
;;       (update-input-value state :password password))
;;     :signup/terms-of-service
;;     (fn [state terms-of-service]
;;       (update-input-value state :terms-of-service terms-of-service))}))

(defn merge-spots [state spots]
  (reduce
   (fn [state spot]
     (update-in state [(:id spot)] #(deep-merge % spot)))
   state spots))

(defn assoc-in-spot
  "Assoc `value` under `ks` in `spot`."
  [state spot ks value]
  (assoc-in state (conj (:id spot) ks) value))

;; (def spot
;;   (coolant/store
;;    :spot {}
;;    {:country/spots
;;     (fn [state {:keys [spots]}]
;;       (merge-spots state spots))
;;     :region/spots
;;     (fn [state {:keys [spots]}]
;;       (merge-spots state spots))
;;     :new-spot/spots
;;     (fn [state spots]
;;       (merge-spots state spots))
;;     :spots/received
;;     (fn [state new-spots]
;;       (merge-spots state new-spots))
;;     :spot/received
;;     (fn [state spot]
;;       (assoc state (:id spot) spot))
;;     :spot/header
;;     (fn [state header]
;;       (assoc state :header header))
;;     :spot/spots-around
;;     (fn [state [spot spots-around]]
;;       (-> (assoc-in state [(:id spot) :spots-around] (map :id spots-around))
;;           (merge-spots spots-around)))
;;     :spot/like-photo
;;     (fn [state [spot photo]]
;;       (assoc-in state [(:id spot) :_embedded :photo] photo))
;;     :spot/dislike-photo
;;     (fn [state [spot photo]]
;;       (assoc-in state [(:id spot) :_embedded :photo] photo))
;;     :spot/weather
;;     (fn [state [spot weather]]
;;       (update-in state [(:id spot) :_embedded :weather] #(merge % weather)))
;;     :spot/weather-time
;;     (fn [state {:keys [spot time]}]
;;       (assoc-in state [(:id spot) :weather-time] time))
;;     :spot/wave-heights-chart
;;     (fn [state {:keys [spot chart]}]
;;       (assoc-in state [(:id spot) :_embedded :wave-heights] chart))}))

;; (def spot-list
;;   (coolant/store
;;    :spots {:list []}
;;    {:spots/received
;;     (fn [state spots]
;;       (update-in state [:list] concat (map :id spots)))}))

(def spots-by-country
  (coolant/store
   :spots-by-country {}
   {:country/spots
    (fn [state {:keys [country spots]}]
      (update-in state [(:id country)] concat (map :id spots)))}))

(def spots-by-region
  (coolant/store
   :spots-by-region {}
   {:region/spots
    (fn [state {:keys [region spots]}]
      (update-in state [(:id region)] concat (map :id spots)))}))

(def session
  (coolant/store
   :session {:rating nil
             :spot nil
             :time nil
             :wave-heights nil
             :weather nil}
   {:session/created
    (fn [state {:keys [spot weather wave-heights]}]
      (assoc state :open false :rating nil))
    :session/open-dialog
    (fn [state {:keys [spot weather wave-heights]}]
      (assoc state
             :open true
             :spot spot
             :time (first (sort (keys weather)))
             :wave-heights wave-heights
             :weather weather
             :time-zone (hal/embedded spot :time-zone)))
    :session/change-time
    (fn [state time]
      (assoc state :time time))
    :session/change-rating
    (fn [state rating]
      (-> (assoc state :rating rating)
          (update-in [:errors] dissoc :rating)))
    :session/close-dialog
    (fn [state _]
      (assoc state :open false))
    :session/submit
    (fn [state {:keys [spot session]}]
      (assoc-in state [(:id spot) :session :submitted-at] (js/Date.)))
    :session/unprocessable
    (fn [state {:keys [spot errors]}]
      (-> (reduce-kv
           (fn [state key error]
             (->> {:value (get-in state [key :value])
                   :error error}
                  (errors/message-for (keyword "session" (name key)))
                  (assoc-in state [:errors key])))
           state errors)
          (assoc :failed-at (js/Date.))))}))

(defn current-location []
  (or (:location hodgepodge/local-storage)
      (geo/point 4326 115.0843 -8.8287)))

(def new-spot
  (coolant/store
   :new-spot (let [location (current-location)]
               {:name {:value nil}
                :address {:value nil}
                :latitude {:value (geo/point-y location)}
                :longitude {:value (geo/point-x location)}
                :photos []})
   {:new-spot/address
    (fn [state address]
      (assoc state :address address))
    :new-spot/bounding-box
    (fn [state bounding-box]
      (assoc state :bounding-box bounding-box))
    :new-spot/latitude
    (fn [state latitude]
      (let [error (errors/message-for :latitude {:value latitude})]
        (assoc state :latitude {:value latitude :error error})))
    :new-spot/longitude
    (fn [state longitude]
      (let [error (errors/message-for :longitude {:value longitude})]
        (assoc state :longitude {:value longitude :error error})))
    :new-spot/name
    (fn [state name]
      (update-input-value state :name name))
    :new-spot/submitted-at
    (fn [state submitted-at]
      (update-input-value state :submitted-at submitted-at))
    :new-spot/spots
    (fn [state spots]
      (assoc state :spots (map :id spots)))
    :new-spot/created
    (fn [state spot]
      (assoc state :name {}))
    :new-spot/unprocessable
    (fn [state {:keys [data errors]}]
      (-> (reduce-kv
           (fn [state key error]
             (->> {:value (get-in state [key :value])
                   :error error}
                  (errors/message-for (keyword "new-spot" (name key)))
                  (assoc-in state [key :error])))
           state errors)
          (assoc :failed-at (js/Date.))))
    :new-spot/photos
    (fn [state photos]
      (assoc state :photos photos))}))

(def toast
  (coolant/store
   :toast {:duration 3000
           :visible false
           :text nil}
   {:session/created
    (fn [state _]
      (assoc state
             :text "Thanks for your rating."
             :visible true
             :visible-at (js/Date.)))}))

(def layout
  (coolant/store
   :layout {:scroller nil}
   {:layout/scroller
    (fn [state scroller]
      (assoc state :scroller scroller))}))

(defn coolant []
  (coolant/core
   [auto-complete/store
    countries/store
    country/store
    layout
    map/store
    new-spot
    page/store
    region/store
    regions/store
    resources/store
    route/store
    search/store
    session
    signin/store
    signup/store
    spot/store
    spots-by-country
    spots-by-region
    spots/store
    toast
    ;; user
    weather/store]))
