(ns burningswell.web.ui.google-map
  (:require [clojure.string :as str]
            [geo.core :as geo]
            [rum.core :as rum]))

(defn point->map
  [point]
  {:latitude (.lat point)
   :longitude (.lng point)})

(defn point->lat-lng
  "Convert `point` to a Google Maps compatible location object."
  [location]
  #?(:cljs (when location
             #js {:lat (-> location :coordinates second)
                  :lng (-> location :coordinates first)})))

(defn lat-lng->point
  "Convert `point` to a Google Maps compatible location object."
  [location]
  (geo/point 4326 (.lng location) (.lat location)))

(defn center
  "Returns the bounding box of `component`."
  [map]
  (lat-lng->point (.getCenter map)))

(defn bounding-box
  "Returns the bounding box of `map` as a geo box."
  [map]
  (let [bounds (.getBounds map)]
    (geo/bounding-box
     (lat-lng->point (.getSouthWest bounds))
     (lat-lng->point (.getNorthEast bounds)))))

(defn bounding-box-query-params
  "Returns the bounding box of `map` as query params."
  [map]
  (let [bounds (.getBounds map)]
    {:top (.lat (.getNorthEast bounds))
     :right (.lng (.getNorthEast bounds))
     :bottom (.lat (.getSouthWest bounds))
     :left (.lng (.getSouthWest bounds))}))

(defn zoom
  "Returns the zoom level of `map`."
  [map]
  (.getZoom map))

(defn event-name
  "Return the Google Maps event name."
  [handler]
  (-> (name handler)
      (str/replace #"^on-" "")
      (str/replace "-" "_")))

(defn map-element
  "Return the map element."
  [state]
  (rum/ref-node state "map"))

(defn map-opts
  "Return the map options"
  [opts]
  #?(:cljs (let [location (or (:location opts) (geo/point 4326 0 0))]
             #js {:center (point->lat-lng location)
                  :zoom (or (:zoom opts) 4)})))

(defn make-map
  "Make a new Google Maps map."
  [state]
  #?(:cljs (google.maps.Map.
            (map-element state)
            (map-opts (-> state :rum/args first)))))

(defn event-handlers
  "Return the Google Maps event name."
  [state]
  (->> (for [[handler callback] (-> state :rum/args first)
             :when (and (keyword? handler)
                        (re-matches #"^on-.+" (name handler)))]
         [handler callback])
       (into {})))

(defn register-event-handlers
  "Register the event handlers for `map`."
  [state]
  (doseq [[handler callback] (event-handlers state)]
    (.addListener (::map state) (event-name handler) #(apply callback (::map state) %)))
  state)

(defn options
  "Return the options from `state`."
  [state]
  (-> state :rum/args first))

(defn location
  "Return the location from `state`."
  [state]
  (-> state :rum/args first :location))

(defn set-center!
  "Set the map center to `location`."
  [state location]
  (.setCenter (::map state) (point->lat-lng location)))

(def google-map-mixin
  "The google-map mixin."
  {:did-mount
   (fn [state]
     (->> (assoc state ::map (make-map state))
          (register-event-handlers)))
   :did-remount
   (fn [old-state new-state]
     ;; (when-not (= (options old-state) (options new-state))
     ;;   (google.maps.event.clearInstanceListeners (::map old-state))
     ;;   (register-event-handlers new-state))
     (when-not (= (location old-state) (location new-state))
       (set-center! new-state (location new-state)))
     new-state)
   :will-unmount
   (fn [state]
     (dissoc state ::map))})

(rum/defc google-map < rum/static google-map-mixin
  "Render an image."
  [opts]
  [:div {:class (:class opts) :ref "map"}])
