(ns burningswell.web.modules.signup
  "The signup page."
  (:require #?(:cljs [burningswell.web.actions :as actions])
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.signup :as signup]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.ui.image :refer [image]]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.links :as link-to]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(rum/defc email < rum/static
  "Render the email form field."
  [system value error]
  (mdl/textfield
   {:class "signup__email"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:id "signup__email"
       :on-change #?(:cljs (actions/on-signup-email-changed system) :clj nil)
       :value value})
     "signup-email")
   (mdl/textfield-label
    {:for "signup__email"}
    "Your email address")
   (mdl/textfield-error error)))

(rum/defc username < rum/static
  "Render the username form field."
  [system value error]
  (mdl/textfield
   {:class "signup__username"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:id "signup__username"
       :on-change #?(:cljs (actions/on-signup-username-changed system) :clj nil)
       :value value})
     "signup-username")
   (mdl/textfield-label
    {:for "signup__username"}
    "Choose your username")
   (mdl/textfield-error error)))

(rum/defc password < rum/static
  "Render the password form field."
  [system value error]
  (mdl/textfield
   {:class "signup__password"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:id "signup__password"
       :on-change #?(:cljs (actions/on-input-changed system :signup/password) :clj nil)
       :type "password"
       :value value})
     "signup-password")
   (mdl/textfield-label
    {:for "signup__password"}
    "Choose a secure password")
   (mdl/textfield-error error)))

(rum/defc submit-button < rum/static
  "Render the signup button."
  [system]
  (mdl/button
   {:class "signup__submit"
    :mdl [:colored :raised :ripple]
    :on-click (fn [e] (.preventDefault e))}
   "Create Account"))

(rum/defc facebook-signup < rum/static
  "Render the signup with Facebook button."
  [system]
  (mdl/button
   {:class "signup__facebook"
    :mdl [:colored :raised :ripple]
    :on-click (fn [e] (.preventDefault e))}
   "Sign in with Facebook"))

(rum/defc help < rum/static
  "Render the signup help."
  [system]
  [:div.signup__member
   "Already a member?" (link-to/login system)])

(rum/defc signup-form < rum/static
  "Render the signup form."
  [system form errors]
  [:form.signup__form
   (email system (:email form) (:email errors))
   (username system (:username form) (:username errors))
   (password system (:password form) (:password errors))
   (submit-button system)
   [:div.signup__or [:span "or"]]
   (facebook-signup system)
   (help system)])

(rum/defc signup-card < rum/static
  "Render the signup card."
  [system form errors]
  (mdl/card
   {:class "signup__card"}
   (mdl/card-title "Sign Up")
   (mdl/card-text "Create a new account.")
   (signup-form system form errors)))

(defn signup-background
  "Render the background image of the signup page."
  []
  (image
   {:class "signup__image"
    :background "black"
    :preload true
    :sizing "cover"
    :fade true
    :src "https://farm9.staticflickr.com/8113/8683411065_e546bcdd3b_o.jpg"}))

(rum/defc content < rum/static
  "Render the content of the signup page."
  [system {:keys [errors form] :as page}]
  (layout
   system page
   [:div.signup__content
    (signup-background)
    (signup-card system form errors)]))

(rum/defcs page < (coolant/mixin signup/page)
  "Render the signup page."
  [page system]
  (content system page))

(defmethod render-server :signup [system]
  #?(:clj (->> (coolant/get system signup/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :signup)
