(ns burningswell.web.ui.weather.table
  (:require #?(:cljs [goog.format :as format])
            [burningswell.web.time :as time]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(def wind-directions
  ["North" "N/E" "East" "S/E" "South" "S/W" "West"])

(defn format-number
  "Format the decimal number `n`."
  [n & [decimals]]
  #?(:clj (format "%.2f" n) :cljs (format/numericValueToString n decimals)))

(defn variable [weather name timestamp & [decimals]]
  (let [{:keys [value unit]} (get-in weather [name timestamp])]
    [:span {:class name}
     [:span.value (format-number value decimals)] " "
     [:span.unit unit]]))

(defn direction [degree]
  (when (number? degree)
    (let [idx (mod (long (/ degree 45)) (count wind-directions))]
      (nth wind-directions idx))))

;; (defn select-weather-time [weather time time-zone]
;;   (->> (filter #(>= % time) (sort (keys weather)))
;;        (identity)
;;        ;; (filter #(time/day-time? % time-zone))
;;        ))

(defn select-weather-time [weather time time-zone]
  (sort (keys (:windsfc weather))))

(rum/defc weather-table < rum/static
  "Render a weather forecast as a table."
  [weather & [opts]]
  (let [{:keys [class time time-zone]} opts]
    (let [class (or class "weather-table")
          timestamps (select-weather-time weather time time-zone)
          time-zone (time/zone-by-offset (:offset time-zone))]
      (when-not (empty? timestamps)
        [:div {:class class}
         [:table
          {:class (str class "__table")}
          [:thead
           {:key "header"}
           [:tr
            [:th.time
             (mdl/icon "schedule")
             "Time"]

            [:th.wave {:col-span 3}
             (mdl/icon "equalizer")
             "Waves"]

            [:th.wind {:col-span 2}
             (mdl/icon "flag")
             "Wind"]

            [:th.temp
             (mdl/icon "opacity")
             "Temp."]]]
          [:tbody
           {:key "body"}
           (for [timestamp timestamps]
             [:tr {:key (.getTime timestamp)}
              [:td.time
               [:div {:class (str class "__time")}
                (time/format-time timestamp time-zone)]
               [:div {:class (str class "__date")}
                (time/format-date timestamp time-zone)]]

              [:td.wave
               [:div {:class (str class "__wave-height-value")}
                (variable weather :htsgwsfc timestamp 1)]
               [:div {:class (str class "__wave-height-description")}
                "Height"]]

              [:td.perpwsfc
               [:div {:class (str class "__wave-period-value")}
                (variable weather :perpwsfc timestamp)]
               [:div {:class (str class "__wave-period-description")}
                "Period"]]

              [:td.dirpwsfc
               [:div {:class (str class "__wave-direction-value")}
                (direction (get-in weather [:dirpwsfc timestamp :value]))]
               [:div {:class (str class "__wave-direction-description")}
                "Direction"]]

              [:td.windsfc
               [:div {:class (str class "__wind-speed-value")}
                (variable weather :windsfc timestamp)]
               [:div {:class (str class "__wind-speed-description")}
                "Speed"]]

              [:td.wdirsfc
               [:div {:class (str class "__wind-direction-value")}
                (direction (get-in weather [:wdirsfc timestamp :value]))]
               [:div {:class (str class "__wind-direction-description")}
                "Direction"]]

              [:td.temp
               [:div {:class (str class "__temperature-value")}
                (some-> (get-in weather [:tmpsfc timestamp :value]) int)
                " °" (get-in weather [:tmpsfc timestamp :unit])]
               [:div {:class (str class "__temperature-description")}
                "Air Temp."]]])]]
         (when time-zone
           [:div {:class (str class "__time-zone-info")}
            (str "Time is shown in the surf spot's local time zone, "
                 ;; TODO: Implement for Clojure.
                 #?(:cljs (.getLongName time-zone time)))])]))))
