(ns burningswell.web.signup-test
  (:require [burningswell.web.signin-test :as signin]
            [burningswell.web.test :refer :all]
            [burningswell.web.taxi :as taxi]
            [clojure.test :refer :all]))

(def signup-data
  {:email "john.doe@example.com"
   :username "johndoe"
   :password "secret"
   :terms-of-service true})

(defn fill-form
  "Fill the signup form."
  [system {:keys [email username password terms-of-service]}]
  (taxi/quick-fill
   (:webdriver system)
   {"input[name=email]" email}
   {"input[name=username]" username}
   {"input[name=password]" password}
   ;; {"input[name=terms-of-service]"
   ;;  #(if terms-of-service (taxi/click (:webdriver system) %))}
   ))

(defn submit-form
  "Fill and submit the signup form."
  [system credentials]
  (browse-to system "/signup")
  (fill-form system credentials)
  (taxi/submit (:webdriver system) "input[name=email]"))

(defn form-field-error
  "Return the error message of the input `field`."
  [system key]
  (->> (format ".signup__%s .mdl-textfield__error" (name key))
       (taxi/text (:webdriver system))))

(defn error
  [system key expected]
  (taxi/wait-until
   (:webdriver system)
   (fn [driver] (= (form-field-error system key) expected))
   10000 100))

(deftest test-signup-submit-sucess
  (with-webdriver-system [system env]
    (submit-form system signup-data)
    (is (wait-for-path (:webdriver system) #"/"))
    ;; (signin/submit-signin-form
    ;;  system {:login (:username signup-data)
    ;;          :password (:password signup-data)})
    ;; (wait-for-path (:webdriver system) #"/spots")
    (click-route system :signout)
    (click-route system :signup)
    (doseq [input [:email :username :password]]
      (is (= (form-field-value system input) ""))
      (is (= (form-field-error system input) "")))))

(deftest test-signup-submit-failed
  (with-webdriver-system [system env]
    (submit-form system {:email "a" :username "b" :password "c"})
    (is (error system :email "Invalid email address."))
    (is (error system :password "Password is too short (min. 4 chars)."))
    (is (error system :username "Username is too short (min. 2 chars)."))))

(deftest test-signup-username-blank
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :username " "))
    (is (error system :username "Username can't be blank."))))

(deftest test-signup-username-too-short
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :username "x"))
    (is (error system :username "Username is too short (min. 2 chars)."))))

(deftest test-signup-username-taken
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :username "bodhi"))
    (is (error system :username "That username is already being used."))))

(deftest test-signup-email-blank
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :email " "))
    (is (error system :email "Email can't be blank."))))

(deftest test-signup-email-invalid
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :email "invalid"))
    (is (error system :email "Invalid email address."))))

(deftest test-signup-email-taken
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :email "bodhi@burningswell.com"))
    (is (error system :email "That email is already being used."))))

(deftest test-signup-password-blank
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :password " "))
    (is (error system :password "Password can't be blank."))))

(deftest test-signup-password-too-short
  (with-webdriver-system [system env]
    (browse-to system "/signup")
    (fill-form system (assoc signup-data :password "x"))
    (is (error system :password "Password is too short (min. 4 chars)."))))

;; (deftest test-signup-terms-of-service-not-accepted
;;   (with-webdriver-system [system env]
;;     (->> (assoc signup-data :terms-of-service false)
;;          (submit-form system))
;;     (is (= (form-field-error system :terms-of-service)
;;            "Please accept the Terms of Service."))))
