(ns burningswell.web.modules.spot
  (:require [#?(:clj clj-time.core :cljs cljs-time.core) :as time]
            [#?(:clj clj-time.coerce :cljs cljs-time.coerce) :as coerce]
            [burningswell.web.actions :as actions]
            [burningswell.web.coolant :as coolant]
            ;; [burningswell.web.time :as time]
            [burningswell.web.getter.spot :as spot]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.spot-header :refer [spot-header]]
            [burningswell.web.ui.spot-list :refer [spot-list]]
            [burningswell.web.ui.weather.table :refer [weather-table]]
            [geo.core :refer [distance-to]]
            [rum.core :as rum]))

(defn distance-between
  "Calculate the distance in km between `spot-1` and `spot-2`."
  [spot-1 spot-2]
  (distance-to (:location spot-1) (:location spot-2)))

(defn assoc-distance-between
  "Assoc onto `spots` the distance in km between them and `spot`."
  [spot spots]
  (map #(assoc % :distance (distance-between spot %)) spots))

(defn previous-weather
  [system spot {:keys [start end]}]
  (fn [event]
    (actions/spot-weather
     system spot
     {:start (time/minus (coerce/to-date-time start) (time/days 1))
      :end start})))


(defn next-weather
  [system spot {:keys [start end]}]
  (fn [event]
    (actions/spot-weather
     system spot
     {:start end
      :end (time/plus (coerce/to-date-time end) (time/days 1))})))

(rum/defc spots-around < rum/static
  "Render the `spots` that are around `spot`."
  [system spot spots]
  (when-not (empty? spots)
    [:div.spot__spots-around
     [:h2.spot__spots-around-headline "Nearby Spots"]
     (spot-list system (assoc-distance-between spot spots))]))

(rum/defc spot-weather < rum/static
  "Render the weather forecast for `spot`."
  [system spot time-range weather]
  (when-not (empty? weather)
    [:div.spot__weather
     [:h2.spot__weather-headline "Surf Forecast"]
     (weather-table
      weather
      {:on-previous (previous-weather system spot time-range)
       :on-next (next-weather system spot time-range)})]))

(rum/defc content < rum/static
  "Render the content of the spot page."
  [system {:keys [spot spots photos time-range weather] :as page}]
  (layout
   system page
   [:div.spot__content
    (spot-header system spot photos)
    [:div.spot__details
     (spot-weather system spot time-range weather)
     (spots-around system spot spots)]]))

(rum/defcs page < (coolant/mixin spot/page)
  "Render the spot page."
  [page system]
  (content system page))

(defmethod render-server :spot [system]
  #?(:clj (->> (coolant/get system spot/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :spot)
