(ns burningswell.web.ui.weather.table
  (:require #?(:cljs [goog.format :as format])
            [burningswell.web.time :as time]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(def wind-directions
  ["North" "N/E" "East" "S/E" "South" "S/W" "West"])

(defn format-number
  "Format the decimal number `n`."
  [n & [decimals]]
  #?(:clj (format "%.2f" n) :cljs (format/numericValueToString n decimals)))

(defn variable [weather name & [decimals]]
  (let [{:keys [value unit]} (get weather name)]
    [:span {:class name}
     [:span.value (format-number value decimals)] " "
     [:span.unit unit]]))

(defn direction [degree]
  (when (number? degree)
    (let [idx (mod (long (/ degree 45)) (count wind-directions))]
      (nth wind-directions idx))))

(rum/defc header < rum/static
  "Render the header of the weather table."
  [weather & [opts]]
  [:thead
   {:key "header"}
   [:tr
    [:th.time
     (mdl/icon "schedule")
     "Time"]
    [:th.wave {:col-span 3}
     (mdl/icon "equalizer")
     "Waves"]
    [:th.wind {:col-span 2}
     (mdl/icon "flag")
     "Wind"]
    [:th.temp
     (mdl/icon "opacity")
     "Temp."]]])

(rum/defc body < rum/static
  "Render the body of the weather table."
  [weather & [opts]]
  (let [{:keys [class time time-zone]} opts]
    (let [class (or class "weather-table")
          time-zone (time/zone-by-offset (:offset time-zone))]
      [:tbody
       {:key "body"}
       (for [weather weather
             :let [timestamp (-> weather :windsfc :valid-time)]
             :when timestamp]
         [:tr {:key (.getTime timestamp)}
          [:td.time
           [:div {:class (str class "__time")}
            (time/format-time timestamp time-zone)]
           [:div {:class (str class "__date")}
            (time/format-date timestamp time-zone)]]

          [:td.wave
           [:div {:class (str class "__wave-height-value")}
            (variable weather :htsgwsfc 1)]
           [:div {:class (str class "__wave-height-description")}
            "Height"]]

          [:td.perpwsfc
           [:div {:class (str class "__wave-period-value")}
            (variable weather :perpwsfc)]
           [:div {:class (str class "__wave-period-description")}
            "Period"]]

          [:td.dirpwsfc
           [:div {:class (str class "__wave-direction-value")}
            (direction (get-in weather [:dirpwsfc :value]))]
           [:div {:class (str class "__wave-direction-description")}
            "Direction"]]

          [:td.windsfc
           [:div {:class (str class "__wind-speed-value")}
            (variable weather :windsfc)]
           [:div {:class (str class "__wind-speed-description")}
            "Speed"]]

          [:td.wdirsfc
           [:div {:class (str class "__wind-direction-value")}
            (direction (get-in weather [:wdirsfc :value]))]
           [:div {:class (str class "__wind-direction-description")}
            "Direction"]]

          [:td.temp
           [:div {:class (str class "__temperature-value")}
            (some-> (get-in weather [:tmpsfc :value]) int)
            " °" (get-in weather [:tmpsfc :unit])]
           [:div {:class (str class "__temperature-description")}
            "Air Temp."]]])])))

(rum/defc footer < rum/static
  "Render the footer of the weather table."
  [weather & [opts]]
  [:tfoot
   [:tr
    [:td {:col-span 7}
     (mdl/button
      {:class "weather-table__previous"
       :mdl [:icon]
       :on-click (:on-previous opts)}
      (mdl/icon "keyboard_arrow_left"))
     (mdl/button
      {:class "weather-table__next"
       :mdl [:icon]
       :on-click (:on-next opts)}
      (mdl/icon "keyboard_arrow_right"))]]])

(rum/defc weather-table < rum/static
  "Render a weather forecast as a table."
  [weather & [opts]]
  (let [{:keys [class time time-zone]} opts]
    (let [class (or class "weather-table")
          time-zone (time/zone-by-offset (:offset time-zone))]
      (when-not (empty? weather)
        [:div {:class class}
         [:table
          {:class (str class "__table")}
          (header weather opts)
          (body weather opts)
          (footer weather opts)]
         (when time-zone
           [:div {:class (str class "__time-zone-info")}
            (str "Time is shown in the surf spot's local time zone, "
                 ;; TODO: Implement for Clojure.
                 #?(:cljs (.getLongName time-zone time)))])]))))
