(ns burningswell.web.mixins.spot-map
  (:require [burningswell.web.core :as core]
            [cljs.core.async :as async :refer [<! timeout]]
            [geo.core :as geo]
            [om.core :as om]
            [om-tools.core :refer-macros [defcomponentk]]
            [om-tools.mixin :refer-macros [defmixin]]
            [sablono.core :as html :refer-macros [html]])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn get-google-map
  "Return the Google map of `owner`."
  [owner]
  (.-map owner))

(defn spot-marker-opts
  "Return the spot marker options for `map`."
  [map spot]
  #js {:map map
       :position (core/point->lat-lng (:location spot))
       :title (:name spot)})

(defn add-spot-marker
  "Add a marker for `spot` to `map`."
  [map spot]
  (google.maps.Marker. (spot-marker-opts map spot)))

(defn add-spot-markers
  "Add markers for all `spots` to `map`."
  [owner spots]
  (doall (map #(add-spot-marker (get-google-map owner) %) spots)))

(defmulti handle-message (fn [owner type & args] type))

(defmethod handle-message :pan-to [owner type location]
  (some->> (core/point->lat-lng location)
           (.panTo (get-google-map owner))))

(defmethod handle-message :default [owner type & args]
  (println "spot-map: Unknown message type: " type))

(defn init-channel
  "Initialize the Google map."
  [owner]
  (when-let [channel (om/get-state owner :channel)]
    (go (loop []
          (when-let [msg (<! channel)]
            (apply handle-message owner msg)
            (recur))))))

(defn replace-spot-markers
  "Replace all spot markers on `map`."
  [owner spots]
  (doseq [marker (om/get-state owner :spot-markers)]
    (.setMap marker nil))
  (->> (add-spot-markers owner spots)
       (om/set-state! owner :spot-markers)))

(defn register-idle-handlers
  "Register on idle handlers."
  [owner]
  (when-let [on-idle (om/get-state owner :on-idle)]
    (let [map (get-google-map owner)]
      (google.maps.event.addListener map "idle" #(on-idle map)))))

(defmixin spot-map-mixin
  "The spot map mixin."
  (did-mount [owner]
    (init-channel owner)
    (register-idle-handlers owner))
  (will-receive-props [owner next-props]
    (when-not (= (:spot (om/get-props owner))
                 (:spot next-props))
      (let [map (get-google-map owner)]
        (add-spot-marker map (:spot next-props))
        (some->> (-> next-props :spot :location)
                 (core/point->lat-lng)
                 (.setCenter map))))
    (when-not (= (:spots (om/get-props owner))
                 (:spots next-props))
      (replace-spot-markers owner (:spots next-props)))))
