(ns burningswell.web.components.text-field
  (:require [clojure.string :as str]
            [goog.events.KeyCodes :as KeyCodes]
            [om.core :as om :include-macros true]
            [om-tools.core :refer-macros [defcomponentk]]
            [sablono.core :as html :refer-macros [defhtml html]]))

(defn on-key-down
  "Handle key down events."
  [data owner opts event]
  (cond
    (:on-key-down opts)
    ((:on-key-down opts) event)
    (and (:on-enter opts)
         (= KeyCodes/ENTER (.-keyCode event)))
    ((:on-enter opts) event)
    :else nil))

(defn focus
  "Focus the input element."
  [owner]
  (some-> (om/get-node owner "input") .focus))

(defn class-name
  "Return the class name from `opts`."
  [opts]
  (or (:class opts) (:name opts)))

(defn class-prefix
  "Prefix `name`  with the class from `opts`."
  [opts name]
  (some-> (class-name opts) (str name)))

(defn error-class
  "Return the CSS class for the error message."
  [opts]
  (class-prefix opts "__error"))

(defhtml error
  "Return an error message, if there are any."
  [data opts]
  (if-let [error (:error data)]
    [:div {:class (error-class opts)} error]))

(defn label-class
  "Return the CSS class for the input field."
  [opts]
  (class-prefix opts "__label"))

(defhtml label
  "Return a label."
  [opts]
  (if-let [label (:label opts)]
    [:label {:class (label-class opts)} label]))

(defn input-class
  "Return the CSS class for the input field."
  [data opts]
  (->> [(class-prefix opts "__input")
        (if (:error data) (class-prefix opts "__input--error"))]
       (remove nil?)
       (str/join " ")))

(defhtml input
  "Return a input field."
  [data owner type opts]
  [:input
   {:class (input-class data opts)
    :auto-complete (:auto-complete opts)
    :name (:name opts)
    :on-blur #((or (:on-blur opts) identity) %)
    :on-change #((or (:on-change opts) identity) %)
    :on-key-down #(on-key-down data owner opts %)
    :placeholder (:placeholder opts)
    :ref "input"
    :type (name type)
    :value (:value data)}])

(defcomponentk text-field
  "Return a text field input component."
  [data owner opts]
  (did-mount [_]
    (when (:focus opts) (focus owner)))
  (render [_]
    (html
     [:div {:class (class-name opts)}
      (label opts)
      (input data owner :text opts)
      (error data opts)])))
