(ns burningswell.web.cookies
  (:refer-clojure :exclude [get])
  (:require #?(:cljs [goog.net.cookies :as gcookies])
            [burningswell.web.logging :as log]
            [burningswell.transit :as transit]
            [no.en.core :as noencore]))

(def ^:dynamic *defaults*
  "The default cookie options."
  {:max-age (* 60 60 24) :path "/"})

(def logger
  "The logger of the cookies namespace."
  (log/logger "burningswell.web.cookies"))

(defn encode
  "Encode `x` into a cookie."
  [x]
  (some-> (transit/encode x)
          #?(:clj .getBytes)
          noencore/base64-encode
          noencore/url-encode))

(defn decode
  "Decode `x` from a cookie."
  [x]
  (some-> (noencore/base64-decode x)
          #?(:clj (String.))
          transit/decode))

(defn decode-safe
  "Safely decode `x` from a cookie safe form."
  [x]
  (try (decode x)
       (catch #?(:clj Exception :cljs js/Error) e
           (log/error logger (str "Can't decode cookie: " x)))))

(defn get
  "Get a cookie by `key` and decode it."
  [key]
  #?(:cljs (when-let [val (gcookies/get (name key))]
             (decode val))))

(defn set-cookie!
  "Encode and store the cookie `val` under `key`."
  [key val & [{:keys [max-age path domain secure]}]]
  #?(:cljs (gcookies/set (name key) (encode val) max-age path domain secure)))

(defn remove!
  "Remove the cookie stored under `key`."
  [key]
  #?(:cljs (gcookies/remove (name key))))

(defn save-location
  "Save the current `location` to a cookie."
  [location]
  (set-cookie! :location location *defaults*))

(defn save-viewport-size
  "Save the current `location` to a cookie."
  [viewport-size]
  (set-cookie! :viewport-size viewport-size *defaults*))

(defn location
  "Get the location cookie from `request`."
  [request]
  (some-> (get-in request [:cookies "location" :value]) decode-safe))

(defn viewport-size
  "Get the viewport size cookie from `request`."
  [request]
  (some-> (get-in request [:cookies "viewport-size" :value]) decode-safe))
