(ns burningswell.web.image
  (:require [om-tools.core :refer-macros [defcomponentk]]
            [om.core :as om :include-macros true]
            [sablono.core :refer-macros [html]]))

(defn- load-image
  "Load the image from `url`."
  [owner url]
  (let [image (js/Image.)]
    (set! (.-onload image) #(om/set-state! owner :image image))
    (set! (.-src image) url)))

(defn- style-url
  "Return `url` in CSS style."
  [url]
  (if url (str "url(" url ")")))

(defn- image-element
  "Return the image element."
  [data owner]
  (let [{:keys [image preload]} (om/get-state owner)]
    (html
     [:img
      {:ref "img"
       :role "none"
       :src
       (cond
         (and preload image)
         (.-src image)
         (not preload)
         data)}])))

(defn- placeholder-element
  "Return the placeholder element."
  [owner]
  (let [{:keys [fade load image placeholder position sizing]}
        (om/get-state owner)]
    (html
     [:div
      {:ref "placeholder"
       :style
       {:background-color "inherit"
        :background-size sizing
        :background-position (if sizing position)
        :background-repeat (if sizing "no-repeat")
        :background-image
        (cond
          image
          (style-url (.-src image))
          (and load placeholder)
          (style-url placeholder))
        :opacity (if image 1 0)
        :transition (if fade "opacity 1s linear")
        :width "100%"
        :height "100%"}}])))

(defn container-attrs
  "Return the attributes of the container element."
  [data owner]
  (let [{:keys [class height fade image preload position
                sizing width]} (om/get-state owner)]
    {:class (or class "image")
     :style
     {:background-size sizing
      :background-position (if sizing position)
      :background-repeat (if sizing "no-repeat")
      :background-image
      (cond
        (and preload image (not fade))
        (style-url (.-src image))
        (and sizing (not preload) (not fade))
        (style-url data))
      ;; :display "inline-block"
      :height (if height (str height "px"))
      :overflow "hidden"
      :position "relative"
      :width (if width (str width "px"))}}))

(defcomponentk image
  [data owner state opts]
  (init-state [_]
    {:class (:class opts)
     :fade (:fade opts)
     :height (:height opts)
     :load true
     :position (:position opts "center")
     :preload (or (:preload opts) (:fade opts))
     :sizing (:sizing opts)
     :width (:width opts)})
  (will-mount [_]
    (if (:preload @state)
      (load-image owner data)))
  (did-update [this prev-props prev-state]
    (if (and (:preload @state) (not= prev-props data))
      (load-image owner data)))
  (render [_]
    (let [{:keys [fade preload sizing]} @state]
      (html
       [:div
        (container-attrs data owner)
        (if-not sizing
          (image-element data owner))
        (if (and preload fade)
          (placeholder-element owner))]))))
