(ns burningswell.web.modules.signin
  "The signin page."
  (:require [burningswell.web.actions :as actions]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.dom :as dom]
            [burningswell.web.getter.signin :as signin]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.ui.facebook :as facebook]
            [burningswell.web.ui.image :refer [image]]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.links :as link-to]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn on-change [system field]
  (fn [event]
    (->> (dom/value (.-target event))
         (actions/dispatch! system field))))

(defn submit-form
  "Submit the signin form."
  [system form event]
  (actions/signin system form)
  (.preventDefault event))

(rum/defc login < rum/static
  "Render the login form field."
  [system value error]
  (mdl/textfield
   {:mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:id "signin__login"
       :name "login"
       :on-change (on-change system :signin/login)
       :value (str value)})
     "signin-login")
   (mdl/textfield-label
    {:for "signin__login"}
    "Your username or email")))

(rum/defc password < rum/static
  "Render the password form field."
  [system value error]
  (mdl/textfield
   {:mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:id "signin__password"
       :name "password"
       :on-change (on-change system :signin/password)
       :type "password"
       :value (str value)})
     "signin-password")
   (mdl/textfield-label
    {:for "signin__password"}
    "Your password")))

(rum/defc remember-me < rum/static
  "Render the remember me form field."
  [system checked]
  (mdl/checkbox
   {:class "signin__remember-me"
    :label "Keep me logged in on this device"
    :checked (true? checked)
    :on-change (on-change system :signin/remember-me)}))

(rum/defc signin-button < rum/static
  "Render the signin button."
  [system form]
  (mdl/button
   {:class "signin__submit"
    :mdl [:colored :raised :ripple]
    :on-click #(submit-form system form %)}
   "Sign in"))

(rum/defc help < rum/static
  "Render the signin help."
  [system]
  [:div.signin__no-account
   "No account yet?"
   (link-to/create-account system)])

(defn facebook-button
  "Render the Facebook signin button."
  [system]
  (facebook/login-button
   (-> system :config :facebook)
   "Sign in with Facebook"
   {:class "signin__facebook"}))

(rum/defc signin-form < rum/static
  "Render the signin form."
  [system form errors]
  [:form.signin__form
   {:on-submit #(submit-form system form %)}
   (when (not-empty errors)
     [:div.signin__errors (first errors)])
   (login system (:login form) (:login errors))
   (password system (:password form) (:password errors))
   (remember-me system (:remember-me form))
   (signin-button system form)
   [:div.signin__or [:span "or"]]
   (facebook-button system)
   (help system)])

(rum/defc signin-card < rum/static
  "Render the signin form as a card."
  [system page]
  (mdl/card
   {:class ["signin__card" (if (:shake? page) "signin__card--shake")]
    :mdl [:shadow--2dp]}
   (mdl/card-title "Sign In")
   (mdl/card-text "Sign in with your existing account.")
   (signin-form system (:form page) (:errors page))))

(rum/defc signin-background < rum/static
  "Render the background image of the signin page."
  []
  (image
   {:class "signin__image"
    :background "black"
    :preload true
    :sizing "cover"
    :fade true
    :src "https://farm9.staticflickr.com/8113/8683411065_e546bcdd3b_o.jpg"}))

(rum/defc content < rum/static
  "Render the content of the signin page."
  [system page]
  (layout
   system page
   [:div.signin__content
    (signin-background)
    (signin-card system page)]))

(rum/defcs page < (coolant/mixin signin/page)
  "Render the signin page."
  [page system]
  (content system page))

(defmethod render-server :signin [system]
  #?(:clj (->> (coolant/get system signin/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :signin)
