(ns burningswell.web.actions
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [clojure.core.async :refer [<! chan close! timeout #?(:clj go)]]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.local-storage :as local-storage]
            [burningswell.web.logging :as log]
            [burningswell.web.api :as api]
            [burningswell.web.getter.core :as getter]
            [burningswell.web.router :as router]
            [burningswell.web.history :as history]
            [burningswell.web.util :as util]
            [burningswell.web.time :as time]
            [clojure.string :as str]
            [geo.core :as geo]
            [hal.core :as hal]
            [no.en.core :refer [format-query-params]])
  #?(:cljs (:import [goog.date DateTime Interval])))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.actions"))

(defn evaluate
  [system getter]
  (coolant/evaluate system getter))

(defn dispatch!
  [system message-type & [message-value]]
  (coolant/dispatch! system message-type message-value))

(defn input-value
  "Return the value of the input `element`."
  [element]
  (case (keyword (.-type element))
    :checkbox (.-checked element)
    (.-value element)))

(defn on-input-changed
  "Dispatch an input changed event."
  [system msg-type]
  (fn [event]
    (dispatch! system msg-type (input-value (.-target event)) )))

(defn on-signup-email-changed
  "Handle email changes in the signup form."
  [system]
  (fn [event]
    (let [value (input-value (.-target event)) ]
      (dispatch! system :signup/email value)
      (go (let [{:keys [status]} (<! (api/user system {:id value}))]
            (dispatch! system :signup/email-available (= status 404)))))))

(defn on-signup-username-changed
  "Handle username changes in the signup form."
  [system]
  (fn [event]
    (let [value (input-value (.-target event)) ]
      (dispatch! system :signup/username value)
      (go (let [{:keys [status]} (<! (api/user system {:id value}))]
            (dispatch! system :signup/username-available (= status 404)))))))

(defn update-value
  "Dispatch an action to update an input value."
  [system {:keys [event topic]}]
  (let [value (input-value (.-target event))]
    (dispatch! system topic value)))

(defn on-value-change
  "Return an on change handler for an input value."
  [system topic]
  #(update-value system {:event % :topic topic}))

(defn- current-location
  "Return the current location."
  [system]
  (evaluate system getter/current-location))

(defn- current-location-params
  "Return the current location as query params."
  [system]
  (util/location-query-params (current-location system)))

(defn change-current-location
  "Change the current geo location."
  [system location]
  (cookies/save-location location)
  (when-not (current-location system)
    (dispatch! system :current/location location)))

(defn wave-heights-chart
  "Fetch the wave heights chart for `spot`."
  [system spot]
  (go (do ;when-not (get-in state [:spot (:id spot) :_embedded :wave-heights])
        (let [{:keys [status body]} (<! (api/wave-heights-chart system spot))]
          (case status
            200 (dispatch! system :spot/wave-heights-chart {:spot spot :chart body})
            404 nil)))))

(defn wave-heights-charts
  "Fetch the wave heights charts for all `spots`."
  [system spots]
  (doseq [spot spots]
    (wave-heights-chart system spot)))

(defn countries-params
  "Return the query params for the countries page."
  [system & [params]]
  (merge {:min-spots 1}
         (current-location-params system)
         (api/pagination system :collection params)
         params))

(defn countries
  "Fetch the countries."
  [system & [params]]
  (dispatch! system :page/loading true)
  (go (let [params (countries-params system params)
            {:keys [status body]} (<! (api/countries system params))]
        (case status
          200 (dispatch! system :countries/list body))
        (dispatch! system :page/loading false))))

(defn country-spots
  "Fetch the spots of `country`."
  [system [country params]]
  (go (let [spots (:body (<! (api/spots-in-country system country params)))]
        (dispatch! system :country/spots {:country country :spots spots})
        (wave-heights-charts system spots))))

(defn country-regions
  "Fetch the regions in `country`."
  [system country & [params]]
  (go (let [params (merge {:min-spots 1} params)
            regions (:body (<! (api/regions-in-country system country params)))]
        (dispatch! system :country/regions regions))))

(defn country
  "Fetch the country."
  [system country]
  (dispatch! system :page/loading true)
  (dispatch! system :country/country country)
  (go (let [country (:body (<! (api/country system country)))]
        (dispatch! system :country/country country)
        (country-regions system country)
        (dispatch! system :page/loading false))))

(defn history
  "Navigate to a link in the application."
  [system [path & [query-params]]]
  (router/navigate (:router system) path query-params))

(defn navigate
  "Navigate to a link in the application."
  [system url & [query-params]]
  (if-let [route (router/match (:router system) url)]
    (history system [url query-params])
    (log/error logger (str "Can't match route for url: " url))))

(defn navigate!
  "Navigate to `url` with `query-params`."
  [system url & [query-params]]
  (navigate system url query-params))

(defn region-spots
  "Fetch the countries."
  [system region & [params]]
  (go (let [spots (:body (<! (api/spots-in-region system region params)))]
        (dispatch! system :region/spots [region spots])
        (wave-heights-charts system spots))))

(defn region
  "Fetch the region."
  [system region]
  (dispatch! system :page/loading true)
  (dispatch! system :region/region region)
  (go (let [region (:body (<! (api/region system region)))]
        (dispatch! system :region/region region)
        (region-spots system region)
        (dispatch! system :page/loading false))))

(defn regions-params
  "Return the query params for the regions page."
  [system & [params]]
  (merge {:min-spots 1}
         (current-location-params system)
         (api/pagination system :collection)
         params))

(defn regions
  "Fetch the regions."
  [system & [params]]
  (dispatch! system :page/loading true)
  (go (let [params (regions-params system params)
            {:keys [status body]} (<! (api/regions system params))]
        (case status
          200 (dispatch! system :regions/list body))
        (dispatch! system :page/loading false))))

(defn auto-complete
  "Search countries, regions and spots."
  [system query cancel & [params]]
  (some-> cancel close!)
  (if (str/blank? query)
    (dispatch! system :auto-complete/clear)
    (dispatch! system :auto-complete/query query))
  (go (when-not (str/blank? query)
        (let [params (assoc params :query query), opts {:cancel (chan)}]
          (dispatch! system :auto-complete/cancel (:cancel opts))
          (let [{:keys [status body]} (<! (api/search-autocomplete system params opts))]
            (case status
              200 (dispatch! system :auto-complete/results body)
              nil ;; Request canceled
              ))))))

(defn search
  "Search countries, regions and spots."
  [system & [{:keys [query] :as params}]]
  (dispatch! system :search/query query)
  (go (when-not (str/blank? query )
        (let [{:keys [status body]} (<! (api/search-details system params))]
          (case status
            200 (dispatch! system :search/results body)
            nil ;; Request canceled
            )))))

(defn search-nearby
  "Search spots nearby."
  [system]
  (go (let [params (merge {:page 1 :per-page 8} (current-location-params system))
            {:keys [status body]} (<! (api/spots system params))]
        (case status
          200 (dispatch! system :search/nearby body)))))

(defn search-details
  "Search countries, regions and spots."
  [system [params]]
  (go (when-not (str/blank? (:query params))
        (let [request (api/search-details system params)
              {:keys [status body]} (<! request)]
          (case status
            200 (do (dispatch! system :search/details body)
                    (if (empty? body)
                      (search-nearby system)
                      (dispatch! system :search/nearby []))))))))

(defn signin-with-token
  "Fetch and set the current user."
  [{:keys [local-storage] :as system} token]
  (go (let [{:keys [status body]} (<! (api/me system token))
            user (assoc body :auth-token token)]
        (case status
          200 (do (local-storage/save-user local-storage user)
                  (dispatch! system :signin/success user)
                  (log/info logger "Successfully logged in.")
                  true)
          401 (do (local-storage/remove-user local-storage)
                  (dispatch! system :signin/failed)
                  (log/error logger "Login failed. Unauthorized (401).")
                  false)
          422 (do (local-storage/remove-user local-storage)
                  (dispatch! system :signin/failed)
                  (log/error logger "Login failed. Unprocessable Entity (422).")
                  false)))))

(defn signin-query-params
  "Sign a user in using the query params."
  [system]
  (go (let [params (:query-params (evaluate system getter/route))]
        (cond
          (:token params)
          (<! (signin-with-token system (:token params)))
          (:error params)
          (do (dispatch! system :signin/failed {:oauth params})
              false)))))

(defn signin-from-route
  "Sign a user in using the token from the query params of `route`."
  [system route]
  (go (let [params (:query-params route)]
        (cond
          (:token params)
          (<! (signin-with-token system (:token params)))
          (:error params)
          (do (dispatch! system :signin/failed {:oauth params})
              false)
          :else nil))))

(defn signin
  "Sign a user in."
  [system user]
  (go (let [{:keys [status body]} (<! (api/create-jws-token system user))]
        (dispatch! system :signin/submit true)
        (case status
          201 (do (dispatch! system :signin/clear)
                  (when (<! (signin-with-token system (:token body)))
                    (history/set-token! (:history system) "/")))
          401 (do (dispatch! system :signin/errors ["Incorrect credentials. Try again!"])
                  (dispatch! system :signin/shake true)
                  (<! (timeout 1000))
                  (dispatch! system :signin/shake false))))))

(defn signup
  "Sign a user up."
  [system user]
  (go (let [{:keys [status body]} (<! (api/create-user system user))]
        (dispatch! system :signup/submit true)
        (case status
          201 (do (dispatch! system :signup/clear body)
                  (signin system {:login (:username user)
                                  :password (:password user)}))
          422 (do (dispatch! system :signup/errors (:errors body))
                  (dispatch! system :signup/shake true)
                  (<! (timeout 1000))
                  (dispatch! system :signup/shake false))))))


;; Spot

(defn spot-weather [system spot & [{:keys [start end]}]]
  (go (let [start (or start (time/today))
            end (or end (time/plus start (time/days 1)))
            {:keys [status body]} (<! (api/spot-weather
                                       system spot
                                       {:start (time/query-param start)
                                        :end (time/query-param end)}))]
        (dispatch! system :spot/weather [spot body])
        (dispatch! system :spot/time-range [spot start end]))))

(defn spot
  "Fetch a spot."
  [system spot & [opts]]
  (let [start (or (:start opts) (time/today))
        end (or (:end opts) (time/plus start (time/days 1)))]
    (dispatch! system :spot/spot spot)
    (go (dispatch!
         system
         :spot/spot
         (:body (<! (api/spot system spot (api/pagination system :collection)))))
        (dispatch!
         system
         :spot/photos
         [spot (:body (<! (api/spot-photos system spot)))])
        (spot-weather system spot opts)
        (dispatch!
         system
         :spot/spots
         [spot (:body (<! (api/spots-around system spot (api/pagination system :spots-around))))])
        ;; (wave-heights-charts system (conj spots spot))
        (dispatch! system :page/loading false))))

(defn spots-params
  "Return the query params for the spots page."
  [system & [params]]
  (merge (current-location-params system)
         (api/pagination system :collection)
         params))

(defn spots
  "Fetch the spots."
  [system & [params]]
  (dispatch! system :page/loading true)
  (go (let [params (spots-params system params)
            {:keys [status body]} (<! (api/spots system params))]
        (case status
          200 (do (dispatch! system :spots/list body)
                  ;; (wave-heights-charts system body)
                  ))
        (dispatch! system :page/loading false))))

(defn like-spot-photo
  "Like a spot photo."
  [system [spot photo]]
  (go (let [{:keys [status body]} (<! (api/like-photo system photo))]
        (case status
          200 (dispatch! system :spot/like-photo [spot body])
          401 (navigate system "/signin")))))

(defn dislike-spot-photo
  "Dislike a spot photo."
  [system [spot photo]]
  (go (let [{:keys [status body]} (<! (api/dislike-photo system photo))]
        (case status
          200 (dispatch! system :spot/dislike-photo [spot body])
          401 (navigate! system "/signin")))))

(defn open-session-dialog
  "Open the session dialog."
  [system spot]
  #?(:cljs (go (let [opts {:start (.toXmlDateTime (doto (DateTime.) (.add (Interval. 0 0 -7))))
                           :end (.toXmlDateTime (DateTime.))}
                     weather (<! (api/spot-weather system spot opts))
                     wave-heights (<! (api/wave-heights-chart system spot opts))]
                 (if (= 200 (:status weather) (:status wave-heights))
                   (dispatch! system :session/open-dialog
                              {:spot spot
                               :weather (:body weather)
                               :wave-heights (:body wave-heights)})
                   (println "Can't fetch data for session dialog!"))))))

(defn close-session-dialog
  "Close the session dialog."
  [system spot]
  (go (dispatch! system :session/close-dialog {:spot spot})))

(defn submit-session
  "Submit a surf session."
  [system {:keys [spot session]}]
  ;; (go (let [{:keys [status body]} (<! (api/create-session system session))]
  ;;       (case status
  ;;         201 (dispatch! system :session/created {:spot spot :session body})
  ;;         401 (navigate! system "/signin")
  ;;         422 (dispatch! system :session/unprocessable (assoc body :spot spot)))))
  )

(defn load-weather
  "Load the weather forecast for `spot`."
  [system spot]
  ;; (go (let [spots (evaluate system getter/spots)
  ;;           weather (get-in spots [(:id spot) :_embedded :weather])]
  ;;       (if (= 1 (count (keys weather)))
  ;;         (let [{:keys [status body]} (<! (api/spot-weather system spot))]
  ;;           (case status
  ;;             200 (dispatch! system :spot/weather [spot body]))))))
  )

(defn display-weather-at-time
  "Change the weather summary `time` for `spot`."
  [system {:keys [spot time] :as msg}]
  (go (dispatch! system :spot/weather-time msg)))

(defn change-session-time
  "Change the create session dialog `time` for `spot`."
  [system time]
  (go (dispatch! system :session/change-time time)))

(defn change-session-rating
  "Change the create session dialog `time` for `spot`."
  [system rating]
  (go (dispatch! system :session/change-rating rating)))

(defn show-spot-header
  [system {:keys [type] :as msg}]
  (go (dispatch! system :spot/header msg)))

(defn set-scroller
  [system scroller]
  (go (dispatch! system :layout/scroller scroller)))

(defn start [system]
  (go (when-let [user (-> system :local-storage :user)]
        (dispatch! system :user/local-storage user))
      (if (<! (signin-query-params system))
        (println "Signin success")
        (println "Signin failed"))))

(defn logout
  "Logout the current user."
  [system]
  (dispatch! system :user/logout))

;; Map

(defn save-map-center
  "Save the map center to local storage."
  [system center]
  (assoc! (:local-storage system) :map-center center))

(defn save-map-zoom
  "Save the map zoom level to local storage."
  [system zoom]
  (assoc! (:local-storage system) :map-zoom zoom))

(defn change-map-bounding-box
  "Change the bounding box of the map."
  [system bounding-box]
  (dispatch! system :map/bounding-box bounding-box))

(defn change-map-center
  "Change the center of the map."
  [system center]
  (dispatch! system :map/center center)
  (save-map-center system center))

(defn change-map-zoom
  "Change the zoom level of the map."
  [system zoom]
  (dispatch! system :map/zoom zoom)
  (save-map-zoom system zoom))

(defn update-map-query-params
  "Update the map query params."
  [system center zoom]
  (->> (assoc (util/location-query-params center) :zoom zoom)
       (router/path-for :map)
       (history/replace-state! (:history system))))

(defn viewport-size-changed
  "Dispatch an event that the viewport size has changed."
  [system viewport-size]
  (dispatch! system :viewport/size viewport-size)
  (cookies/save-viewport-size viewport-size))

(defn map-idle
  "Fetch countries, regions or spots bases on zoom level."
  [system bounding-box center zoom]
  (dispatch! system :page/loading true)
  (update-map-query-params system center zoom)
  (let [params (merge (util/bounding-box-query-params bounding-box) {:zoom zoom :min-spots 1})]
    (go (let [params (merge (api/pagination system :map) params)
              {:keys [status body]} (<! (api/map system params))]
          (case status
            200 (dispatch! system :map/results body))
          (dispatch! system :page/loading false)))))

;; Routing

(defn route-changed
  "Dispatch a route change event."
  [system route]
  (dispatch! system :route/current route))

(defn change-page
  "Dispatch a page change event."
  [system page]
  (dispatch! system :route/page page))

(defmulti change-route
  (fn [system route] (:name route)))

(defmethod change-route :index [system route]
  (route-changed system route)
  (spots system (:query-params route)))

(defmethod change-route :countries [system route]
  (route-changed system route)
  (countries system (:query-params route)))

(defmethod change-route :country [system route]
  (route-changed system route)
  (country system (:params route)))

(defmethod change-route :map [system route]
  (route-changed system route))

(defmethod change-route :region [system route]
  (route-changed system route)
  (region system (:params route)))

(defmethod change-route :search [system route]
  (route-changed system route)
  (search system (:query-params route)))

(defmethod change-route :signup [system route]
  (route-changed system route))

(defmethod change-route :signout [system route]
  (dispatch! system :user/logout)
  (route-changed system {:name :signin}))

(defmethod change-route :signin [system route]
  (route-changed system route))

(defmethod change-route :spot [system route]
  (route-changed system route)
  (spot system (:params route)))

(defmethod change-route :spots [system route]
  (route-changed system route)
  (spots system (:query-params route)))

(defmethod change-route :regions [system route]
  (route-changed system route)
  (regions system (:query-params route)))
