(ns burningswell.web.router
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go-loop]]))
  (:require [#?(:clj clojure.core.async :cljs cljs.core.async) :refer [<! close! #?(:clj go-loop)]]
            [burningswell.web.bus :as bus]
            [burningswell.web.logging :as log]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [no.en.core :refer [parse-url]]
            [routes.core :as routes #?(:clj :refer :cljs :refer-macros) [defroutes]]))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.router"))

(defroutes routes
  ["/countries" :countries]
  ["/countries/:id" :country]
  ["/map" :map]
  ["/regions" :regions]
  ["/regions/:id" :region]
  ["/search" :search]
  ["/signin" :signin]
  ["/signup" :signup]
  ["/signout" :signout]
  ["/spots" :spots]
  ["/spots/:id" :spot]
  ["/" :spots])

(defn- clean-token
  "Return a clean `token`."
  [token]
  (some-> token (str/replace #"^(/?(#?!/)?)?" "/")))

(defn- normalized-url
  "Return the normalized `url`."
  [url]
  (some-> (if (str/starts-with? (str url) "/")
            {:uri url}
            (parse-url url))
          (update-in [:uri] clean-token)))

(defn- subscribe-navigation [router]
  (->> (bus/subscribe (:bus router) :history/navigate)
       (assoc router :subscription)))

(defn match
  "Find the route matching `url` in `router`."
  [router url]
  (let [url (normalized-url url)]
    (when-let [route (routes/route-matches (:routes router) url)]
      route)))

(defn navigate
  "Use the router history to naviagte to `path` with `query-params`."
  [router path & [query-params]]
  (prn "TODO: route navigate"))

(defn current-url
  "Return the current url."
  []
  #?(:cljs js/window.location.href))

(defn reload!
  "Match the current route."
  [router]
  (let [url (normalized-url (current-url))]
    (if-let [route (routes/route-matches (:routes router) url)]
      (do (bus/publish (:bus router) ::navigate {:route route})
          (log/info logger (str "Route matched. " (pr-str url))))
      (log/warning logger (str "Can't match route. " (pr-str url))))))

(defn- handle-navigation [router]
  (go-loop [event (<! (:subscription router))]
    (if event
      (do (reload! router)
          (recur (<! (:subscription router))))
      (log/info logger "Router subscription loop stopped."))))

(defn start-router
  "Start the router."
  [router]
  (if (:subscription router)
    router
    (let [router (subscribe-navigation router)]
      (handle-navigation router)
      (log/info logger "Router successfully started.")
      router)))

(defn stop-router
  "Stop the router."
  [router]
  (when-let [subscription (:subscription router)]
    (close! subscription))
  (log/info logger "Router successfully stopped.")
  (assoc router :subscription nil))

(defrecord Router [bus subscription]
  component/Lifecycle
  (start [router]
    (start-router router))
  (stop [router]
    (stop-router router)))

(defn router
  "Return a new router."
  [routes & [opts]]
  (-> (assoc opts :routes routes)
      (map->Router)
      (component/using [:bus :history])))
