(ns burningswell.web.actions
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.dom :as dom]
            [burningswell.web.getter.core :as getter]
            [burningswell.web.history :as history]
            [burningswell.web.local-storage :as local-storage]
            [burningswell.web.logging :as log]
            [burningswell.web.router :as router]
            [burningswell.web.system.core :as system :refer [change-route]]
            [burningswell.web.time :as time]
            [burningswell.web.util :as util]
            [clojure.core.async :refer [<! chan close! timeout #?(:clj go)]]
            [clojure.string :as str]
            [geo.core :as geo]
            [hal.core :as hal]
            [no.en.core :refer [format-query-params]])
  #?(:cljs (:import [goog.date DateTime Interval])))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.actions"))

(defn evaluate
  [system getter]
  (coolant/evaluate system getter))

(defn input-value
  "Return the value of the input `element`."
  [element]
  (case (keyword (.-type element))
    :checkbox (.-checked element)
    (.-value element)))

(defn on-input-changed
  "Dispatch an input changed event."
  [system msg-type]
  (fn [event]
    (system/dispatch! system msg-type (input-value (.-target event)) )))

(defn update-value
  "Dispatch an action to update an input value."
  [system {:keys [event topic]}]
  (let [value (input-value (.-target event))]
    (system/dispatch! system topic value)))

(defn on-value-change
  "Return an on change handler for an input value."
  [system topic]
  #(update-value system {:event % :topic topic}))

(defn country-spots
  "Fetch the spots of `country`."
  [system [country params]]
  (go (let [spots (:body (<! (api/spots-in-country system country params)))]
        (system/dispatch! system :country/spots {:country country :spots spots})
        (system/wave-heights-charts system spots))))

(defn history
  "Navigate to a link in the application."
  [system [path & [query-params]]]
  (router/navigate (:router system) path query-params))

(defn navigate
  "Navigate to a link in the application."
  [system url & [query-params]]
  (if-let [route (router/match (:router system) url)]
    (history system [url query-params])
    (log/error logger (str "Can't match route for url: " url))))

(defn navigate!
  "Navigate to `url` with `query-params`."
  [system url & [query-params]]
  (navigate system url query-params))

(defn auto-complete
  "Search countries, regions and spots."
  [system query cancel & [params]]
  (some-> cancel close!)
  (if (str/blank? query)
    (system/dispatch! system :auto-complete/clear)
    (system/dispatch! system :auto-complete/query query))
  (go (when-not (str/blank? query)
        (let [params (assoc params :query query), opts {:cancel (chan)}]
          (system/dispatch! system :auto-complete/cancel (:cancel opts))
          (let [{:keys [status body]} (<! (api/search-autocomplete system params opts))]
            (case status
              200 (system/dispatch! system :auto-complete/results body)
              nil ;; Request canceled
              ))))))

(defn search-nearby
  "Search spots nearby."
  [system]
  (go (let [params (merge {:page 1 :per-page 8} (system/current-location-params system))
            {:keys [status body]} (<! (api/spots system params))]
        (case status
          200 (system/dispatch! system :search/nearby body)))))

(defn search-details
  "Search countries, regions and spots."
  [system [params]]
  (go (when-not (str/blank? (:query params))
        (let [request (api/search-details system params)
              {:keys [status body]} (<! request)]
          (case status
            200 (do (system/dispatch! system :search/details body)
                    (if (empty? body)
                      (search-nearby system)
                      (system/dispatch! system :search/nearby []))))))))

(defn signin-query-params
  "Sign a user in using the query params."
  [system]
  (go (let [params (:query-params (evaluate system getter/route))]
        (cond
          (:token params)
          (<! (system/signin-with-token system (:token params)))
          (:error params)
          (do (system/dispatch! system :signin/failed {:oauth params})
              false)))))

;; Spot

(defn like-spot-photo
  "Like a spot photo."
  [system [spot photo]]
  (go (let [{:keys [status body]} (<! (api/like-photo system photo))]
        (case status
          200 (system/dispatch! system :spot/like-photo [spot body])
          401 (navigate system "/signin")))))

(defn dislike-spot-photo
  "Dislike a spot photo."
  [system [spot photo]]
  (go (let [{:keys [status body]} (<! (api/dislike-photo system photo))]
        (case status
          200 (system/dispatch! system :spot/dislike-photo [spot body])
          401 (navigate! system "/signin")))))

(defn open-session-dialog
  "Open the session dialog."
  [system spot]
  #?(:cljs (go (let [opts {:start (.toXmlDateTime (doto (DateTime.) (.add (Interval. 0 0 -7))))
                           :end (.toXmlDateTime (DateTime.))}
                     weather (<! (api/spot-weather system spot opts))
                     wave-heights (<! (api/wave-heights-chart system spot opts))]
                 (if (= 200 (:status weather) (:status wave-heights))
                   (system/dispatch! system :session/open-dialog
                                     {:spot spot
                                      :weather (:body weather)
                                      :wave-heights (:body wave-heights)})
                   (println "Can't fetch data for session dialog!"))))))

(defn close-session-dialog
  "Close the session dialog."
  [system spot]
  (go (system/dispatch! system :session/close-dialog {:spot spot})))

(defn submit-session
  "Submit a surf session."
  [system {:keys [spot session]}]
  ;; (go (let [{:keys [status body]} (<! (api/create-session system session))]
  ;;       (case status
  ;;         201 (system/dispatch! system :session/created {:spot spot :session body})
  ;;         401 (navigate! system "/signin")
  ;;         422 (system/dispatch! system :session/unprocessable (assoc body :spot spot)))))
  )

(defn load-weather
  "Load the weather forecast for `spot`."
  [system spot]
  ;; (go (let [spots (evaluate system getter/spots)
  ;;           weather (get-in spots [(:id spot) :_embedded :weather])]
  ;;       (if (= 1 (count (keys weather)))
  ;;         (let [{:keys [status body]} (<! (api/spot-weather system spot))]
  ;;           (case status
  ;;             200 (system/dispatch! system :spot/weather [spot body]))))))
  )

(defn display-weather-at-time
  "Change the weather summary `time` for `spot`."
  [system {:keys [spot time] :as msg}]
  (go (system/dispatch! system :spot/weather-time msg)))

(defn change-session-time
  "Change the create session dialog `time` for `spot`."
  [system time]
  (go (system/dispatch! system :session/change-time time)))

(defn change-session-rating
  "Change the create session dialog `time` for `spot`."
  [system rating]
  (go (system/dispatch! system :session/change-rating rating)))

(defn show-spot-header
  [system {:keys [type] :as msg}]
  (go (system/dispatch! system :spot/header msg)))

(defn set-scroller
  [system scroller]
  (go (system/dispatch! system :layout/scroller scroller)))

(defn start [system]
  (go (when-let [user (-> system :local-storage :user)]
        (system/dispatch! system :user/local-storage user))
      (if (<! (signin-query-params system))
        (println "Signin success")
        (println "Signin failed"))))
