(ns burningswell.web.stores.resources
  (:require [coolant.core :as coolant]
            [clojure.pprint :refer [pprint]]
            [no.en.core :refer [deep-merge parse-integer]]))

(defn id
  "Return the :id of `resource` as an integer."
  [resource]
  (some-> resource :id parse-integer))

(defn country-path
  "Return the resource path for `country`."
  [country]
  [:country/by-id (id country)])

(defn photo-path
  "Return the resource path for `photo`."
  [photo]
  [:photo/by-id (id photo)])

(defn region-path
  "Return the resource path for `region`."
  [region]
  [:region/by-id (id region)])

(defn spot-path
  "Return the resource path for `spot`."
  [spot]
  [:spot/by-id (id spot)])

(defn user-path
  "Return the resource path for `user`."
  [user]
  [:user/by-id (id user)])

(defn spot-weather-path
  "Return the resource path for the `spot` weather."
  [spot]
  [:weather/by-spot-id (id spot)])

(defn add-resource [path-fn]
  (fn [state resource]
    (update-in state (path-fn resource) #(deep-merge % resource))))

(defn add-resources [path-fn]
  (let [add (add-resource path-fn)]
    (fn [state resources]
      (reduce add state resources))))

(defn update-resource
  "Update the `resource` in `state` at the path produced by applying
  `resource` to `path-fn`."
  [state resource path-fn]
  (update-in state (path-fn resource) #(deep-merge % resource)))

(defn update-spot
  "Update `spot` in `state`."
  [state spot]
  (update-resource state spot spot-path))

(defn update-photo
  "Update `photo` in `state`."
  [state photo]
  (update-resource state photo photo-path))

(defn update-region-spots
  "Update the `spots` of a region in `state`."
  [state [region spots]]
  (reduce update-spot state spots))

(defn update-spot-spots
  "Update the `spots` around a `spot`."
  [state [region spots]]
  (reduce update-spot state spots))

(defn update-spot-photos
  "Update the `spots` around a `spot`."
  [state [region photos]]
  (reduce update-photo state photos))

(defn update-spot-weather
  "Update the `weather` of `spot`."
  [state [spot weather]]
  (update-in state (spot-weather-path spot) #(deep-merge % weather)))

(defn resource
  "Get a resource from `store` by `path`."
  [store path]
  (get-in store path))

(defn resources
  "Get a resources from `store` by `paths`."
  [store paths]
  (map #(get-in store %) paths))

(def store
  "The resources store."
  (coolant/store
   :resources
   {:country/by-id {}
    :photo/by-id {}
    :region/by-id {}
    :spot/by-id {}
    :user/by-id {}
    :weather/by-spot-id {}}
   {:countries/list (add-resources country-path)
    :country/country (add-resource country-path)
    :country/regions (add-resources region-path)
    :region/region (add-resource region-path)
    :region/spots update-region-spots
    :regions/list (add-resources region-path)
    :spot/photos update-spot-photos
    :spot/spot (add-resource spot-path)
    :spot/spots update-spot-spots
    :spot/weather update-spot-weather
    :spots/list (add-resources spot-path)}))
