(ns burningswell.web.search-test
  (:require [burningswell.web.test :refer :all]
            [burningswell.web.taxi :as taxi]
            [clojure.test :refer :all])
  (:import [org.openqa.selenium Keys]))

(defn search-field
  "Find the search input field."
  [webdriver]
  (taxi/find-element webdriver {:css ".auto-complete__query"}))

(defn slow-input-text
  "Slowly enter `text` into the `element` input field."
  [webdriver element text]
  (doseq [char text]
    (taxi/input-text webdriver element (str char))))

(defn autocomplete-input
  "Enter `text` into the autocomplete input field."
  [webdriver text]
  (let [element (search-field webdriver)]
    (slow-input-text webdriver element text)
    (Thread/sleep 500)
    element))

(defn autocomplete-results
  "Return the autocomplete results."
  [webdriver]
  (some->> {:css ".auto-complete-item__headline"}
           (taxi/find-elements webdriver)
           (map #(taxi/text webdriver %))))

(defn autocomplete-selected
  "Return the current autocomplete selection."
  [webdriver]
  (some->> {:css ".auto-complete__result--highlight .auto-complete-item__headline"}
           (taxi/find-element webdriver)
           (taxi/text webdriver)))

(defn wait-for-autocomplete-selected
  "Return the current autocomplete selection."
  [webdriver expected]
  (taxi/wait-until
   webdriver
   (fn [& args] (= (autocomplete-selected webdriver) expected))
   10000 10)
  (autocomplete-selected webdriver))

(defn search-result-spots
  "Return the spot search results."
  [webdriver]
  (->> {:css ".spot-headline__name"}
       (taxi/find-elements webdriver)
       (map #(taxi/text webdriver %))))

(defn wait-for-results [webdriver]
  (taxi/find-element webdriver {:css ".auto-complete__suggestions"}))

(deftest ^:webdriver test-search-query
  (with-webdriver-system [{:keys [webdriver] :as system} env]
    (browse-to system "/search" {:query "praia"})
    (taxi/find-element webdriver {:css ".search-result"})
    (is (= (set (search-result-spots webdriver))
           #{"Praia do Castello"
             "Praia do Guincho"
             "Praia Grande/Pequena"
             "Praia do Peixe"
             "Praia do Norte"}))))

(deftest ^:webdriver test-search-autocomplete-navigation
  (with-webdriver-system [system env]
    (browse-to system "/search")
    (let [webdriver (:webdriver system)
          element (autocomplete-input webdriver "es")]
      (wait-for-results webdriver)
      (taxi/send-keys webdriver element Keys/ARROW_DOWN)
      (is (wait-for-autocomplete-selected webdriver "Monte Estoril"))
      (taxi/send-keys webdriver element Keys/ARROW_DOWN)
      (is (wait-for-autocomplete-selected webdriver "United States"))
      (taxi/send-keys webdriver element Keys/ARROW_UP)
      (is (wait-for-autocomplete-selected webdriver "Monte Estoril")))))

(deftest ^:webdriver test-search-autocomplete-country
  (with-webdriver-system [system env]
    (browse-to system "/search")
    (let [webdriver (:webdriver system)
          element (autocomplete-input webdriver "spain")]
      (wait-for-results webdriver)
      (taxi/send-keys webdriver element Keys/ARROW_DOWN)
      (taxi/send-keys webdriver element Keys/ENTER)
      (is (wait-for-path webdriver #"/countries/.*"))
      ;; (is (= (country-headline webdriver) "Spain"))
      )))

(deftest ^:webdriver test-search-autocomplete-spot
  (with-webdriver-system [system env]
    (browse-to system "/search")
    (let [webdriver (:webdriver system)
          element (autocomplete-input webdriver "mundaka")]
      (wait-for-results webdriver)
      (taxi/send-keys webdriver element Keys/ARROW_DOWN)
      (taxi/send-keys webdriver element Keys/ENTER)
      (is (wait-for-path webdriver #"/spots/.*"))
      (is (= (spot-headline-name webdriver) "Mundaka")))))

(deftest ^:webdriver test-search-autocomplete-region
  (with-webdriver-system [system env]
    (browse-to system "/search")
    (let [webdriver (:webdriver system)
          element (autocomplete-input webdriver "bali")]
      (wait-for-results webdriver)
      (taxi/send-keys webdriver element Keys/ARROW_DOWN)
      (taxi/send-keys webdriver element Keys/ENTER)
      (is (wait-for-path webdriver #"/regions/.*"))
      ;; (is (= (region-headline webdriver) "Bali, Indonesia"))
      )))

(deftest ^:webdriver test-search-autocomplete-results
  (with-webdriver-system [system env]
    (browse-to system "/search")
    (let [webdriver (:webdriver system)
          element (autocomplete-input webdriver "praia")]
      (wait-for-results webdriver)
      (is (= (set (autocomplete-results webdriver))
             #{"Praia do Castello"
               "Praia do Guincho"
               "Praia Grande/Pequena"
               "Praia do Peixe"
               "Praia do Norte"})))))
