(ns burningswell.web.system.core
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.coolant :as coolant]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.dom :as dom]
            [burningswell.web.local-storage :as local-storage]
            [burningswell.web.logging :as log]
            [burningswell.web.api :as api]
            [burningswell.web.getter.core :as getter]
            [burningswell.web.router :as router]
            [burningswell.web.history :as history]
            [burningswell.web.util :as util]
            [burningswell.web.time :as time]
            [clojure.core.async :refer [<! chan close! timeout #?(:clj go)]]
            [clojure.string :as str]
            [geo.core :as geo]
            [hal.core :as hal]
            [no.en.core :refer [format-query-params]])
  #?(:cljs (:import [goog.date DateTime Interval])))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.system.core"))

(defn evaluate
  [system getter]
  (coolant/evaluate system getter))

(defn current-user
  [system]
  (evaluate system getter/current-user))

(defn logged-in?
  "Returns true if the user is currently logged in, otherwise false."
  [system]
  (some? (current-user system)))

(defn current-location
  "Return the current location."
  [system]
  (evaluate system getter/current-location))

(defn current-location-params
  "Return the current location as query params."
  [system]
  (util/location-query-params (current-location system)))

(defn dispatch!
  [system message-type & [message-value]]
  (coolant/dispatch! system message-type message-value))

(defn change-current-location
  "Change the current geo location."
  [system location]
  (cookies/save-location location)
  (when-not (current-location system)
    (dispatch! system :current/location location)))

(defn change-value [system field]
  (fn [event]
    (some->> (.-target event)
             (dom/value)
             (dispatch! system field))))

(defn navigate-to-url [system url]
  (history/set-token! (:history system) url))

(defn route-changed
  "Dispatch a route change event."
  [system route]
  (dispatch! system :route/current route))

(defmulti change-route
  (fn [system route] (:name route)))

(defn auto-complete
  "Search countries, regions and spots."
  [system query cancel & [params]]
  (some-> cancel close!)
  (if (str/blank? query)
    (dispatch! system :auto-complete/clear)
    (dispatch! system :auto-complete/query query))
  (go (when-not (str/blank? query)
        (let [params (assoc params :query query), opts {:cancel (chan)}]
          (dispatch! system :auto-complete/cancel (:cancel opts))
          (let [{:keys [status body]} (<! (api/search-autocomplete system params opts))]
            (case status
              200 (dispatch! system :auto-complete/results body)
              nil ;; Request canceled
              ))))))

(defn signin-with-token
  "Fetch and set the current user."
  [{:keys [local-storage] :as system} token]
  (go (let [{:keys [status body]} (<! (api/me system token))
            user (assoc body :auth-token token)]
        (case status
          200 (do (local-storage/save-user local-storage user)
                  (dispatch! system :signin/success user)
                  (log/info logger "Successfully logged in.")
                  true)
          401 (do (local-storage/remove-user local-storage)
                  (dispatch! system :signin/failed)
                  (log/error logger "Login failed. Unauthorized (401).")
                  false)
          422 (do (local-storage/remove-user local-storage)
                  (dispatch! system :signin/failed)
                  (log/error logger "Login failed. Unprocessable Entity (422).")
                  false)))))

(defn signin-from-route
  "Sign a user in using the token from the query params of `route`."
  [system route]
  (go (let [params (:query-params route)]
        (cond
          (:token params)
          (<! (signin-with-token system (:token params)))
          (:error params)
          (do (dispatch! system :signin/failed {:oauth params})
              false)
          :else nil))))

(defn wave-heights-chart
  "Fetch the wave heights chart for `spot`."
  [system spot]
  (go (do ;when-not (get-in state [:spot (:id spot) :_embedded :wave-heights])
        (let [{:keys [status body]} (<! (api/wave-heights-chart system spot))]
          (case status
            200 (dispatch! system :spot/wave-heights-chart {:spot spot :chart body})
            404 nil)))))

(defn wave-heights-charts
  "Fetch the wave heights charts for all `spots`."
  [system spots]
  (doseq [spot spots]
    (wave-heights-chart system spot)))

(defn viewport-size-changed
  "Dispatch an event that the viewport size has changed."
  [system viewport-size]
  (dispatch! system :viewport/size viewport-size)
  (cookies/save-viewport-size viewport-size))

(defn login
  "Login from local storage."
  [{:keys [local-storage] :as system}]
  (when-let [user (local-storage/current-user local-storage)]
    (when-let [token (:auth-token user)]
      (go (let [{:keys [status body]} (<! (api/me system token))]
            (if (= status 200)
              (do (dispatch! system :signin/success user)
                  (log/info logger "Successfully logged in from local storage.")
                  true)
              (do (local-storage/remove-user local-storage)
                  (dispatch! system :signin/failed)
                  (log/error logger (str "Login from local storage failed. " status "."))
                  false)))))))
