(ns burningswell.web.js-engine
  (:require [com.stuartsierra.component :as component]
            [schema.core :as s])
  (:import [javax.script ScriptEngine ScriptEngineManager ScriptException]
           [jdk.nashorn.api.scripting NashornException]))

(def ^:dynamic *defaults*
  {:name "nashorn"})

(s/defrecord JSEngine
    [name :- s/Str
     instance :- (s/maybe ScriptEngine)]
  {s/Any s/Any})

(defn- javascript-error [engine script exception]
  (let  [cause (.getCause exception)
         msg (if (instance? NashornException cause)
               (str (.getMessage cause) "\n"
                    (NashornException/getScriptStackString cause))
               "Error while evaluating JavaScript.")]
    (throw (ex-info msg
                    {:engine engine
                     :exception exception
                     :script script}))))

(s/defn ^:always-validate evaluate :- s/Any
  "Evaluate the JavaScript `script` with `engine`."
  [engine :- JSEngine script :- s/Any]
  (try (.eval (:instance engine) script)
       (catch ScriptException e
         (javascript-error engine script e))))

(s/defn ^:always-validate create-engine :- JSEngine
  "Create a new JavaScript engine."
  [engine]
  (let [manager (ScriptEngineManager.)]
    (if-let [instance (.getEngineByName manager (:name engine))]
      (assoc engine :instance instance)
      (throw (ex-info "Can't create Engine engine." {})))))

(s/defn ^:always-validate start-engine :- JSEngine
  "Start the Engine component."
  [engine]
  (if (:instance engine)
    engine (create-engine engine)))

(s/defn ^:always-validate stop-engine :- JSEngine
  "Stop the Engine component."
  [engine]
  (assoc engine :instance nil))

(extend-protocol component/Lifecycle
  JSEngine
  (start [engine]
    (start-engine engine))
  (stop [engine]
    (stop-engine engine)))

(s/defn ^:always-validate new-engine :- JSEngine
  "Return a new JavaScript engine."
  [& [config]]
  (map->JSEngine (merge *defaults* config)))

(defmacro with-js-engine
  [[component-sym config] & body]
  `(let [component# (component/start (new-engine ~config))
         ~component-sym component#]
     (try ~@body
          (finally (component/stop component#)))))
