(ns burningswell.web.coolant
  (:refer-clojure :exclude [get])
  (:require [coolant.core :as coolant]
            [rum.core :as rum]))

(def store coolant/store)

(defn coolant?
  "Return true of `x` is a Coolant instance, otherwise false."
  [x]
  (and x (instance? coolant.core.Core @x)))

(defn- coolant-from-state
  "Return coolant component from `state`."
  [state]
  {:post [(coolant? %)]}
  (-> state :rum/args first :coolant))

(defn- coolant-from-system
  "Return coolant component from `system`."
  [system]
  {:post [(coolant? %)]}
  (:coolant system))

(defn get
  "Evaluate `getter`."
  [{:keys [coolant] :as system} getter]
  {:pre [(coolant? coolant)]}
  (coolant/evaluate coolant getter))

(defn dispatch!
  "Dispatch a message to the registered stores to update
  state. Notifies observers if their watched value has
  changed. Returns updated system."
  [system message-type & [message-value]]
  (->> (coolant/message message-type message-value)
       (coolant/dispatch! (coolant-from-system system))
       (assoc system :coolant)))

(defn evaluate
  "Evaluate `getter` using coolant from `state`."
  [system getter]
  (coolant/evaluate (:coolant system) getter))

(defn mixin
  "Return a Coolant mixin that re-renders when `getter` changes."
  [getter]
  #?(:cljs {:init
            (fn [state props]
              (let [coolant (coolant-from-state state)]
                (merge state (coolant/evaluate coolant getter))))
            :will-mount
            (fn [{:keys [rum/react-component] :as state}]
              (->> (fn [getter-state]
                     (when (.isMounted react-component)
                       (let [previous @(rum/state react-component)
                             next (select-keys previous [::observer :rum/args :rum/react-component])
                             next (merge next getter-state)]
                         (set! (.-state react-component) #js {":rum/state" (volatile! next)})
                         (rum/request-render react-component))))
                   (coolant/observe! (coolant-from-state state) getter)
                   (assoc state ::observer)))
            :will-unmount
            (fn [state]
              (when-let [unregister (::observer state)]
                (unregister))
              (dissoc state ::observer))}))
